<?php
// Fichier AJAX pour la gestion des phases - Version simplifiée et robuste
require_once dirname(__DIR__) . '/includes/db.php';

// Désactiver l'affichage des erreurs
error_reporting(0);
ini_set('display_errors', 0);

// Nettoyer le buffer
if (ob_get_level()) {
    ob_clean();
}

// Headers JSON
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, must-revalidate');

// Vérifier que c'est une requête POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Méthode non autorisée']);
    exit;
}

// Vérifier l'action
if (!isset($_POST['action']) || $_POST['action'] !== 'toggle_phase') {
    echo json_encode(['success' => false, 'message' => 'Action non reconnue']);
    exit;
}

// Récupérer et valider les données
$phase_name = trim($_POST['phase_name'] ?? '');
$page_type = trim($_POST['page_type'] ?? '');
$is_active = intval($_POST['is_active'] ?? 0);

// Validation des données
if (empty($phase_name) || empty($page_type)) {
    echo json_encode(['success' => false, 'message' => 'Données manquantes']);
    exit;
}

if (!in_array($page_type, ['quiz', 'gagnants', 'challenge_photos'])) {
    echo json_encode(['success' => false, 'message' => 'Type de page invalide']);
    exit;
}

// Vérifier la connexion à la base de données
if (!$mysqli || $mysqli->connect_error) {
    echo json_encode(['success' => false, 'message' => 'Erreur de connexion à la base de données']);
    exit;
}

try {
    // Vérifier si la phase existe
    $check_stmt = $mysqli->prepare("SELECT id, is_active FROM phase_management WHERE phase_name = ? AND page_type = ?");
    $check_stmt->bind_param('ss', $phase_name, $page_type);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    $check_row = $check_result->fetch_assoc();
    $check_stmt->close();
    
    if ($check_row) {
        // La phase existe, mettre à jour
        if ($check_row['is_active'] == $is_active) {
            echo json_encode(['success' => true, 'message' => 'Phase déjà dans l\'état demandé']);
        } else {
            $stmt = $mysqli->prepare("UPDATE phase_management SET is_active = ? WHERE phase_name = ? AND page_type = ?");
            if (!$stmt) {
                throw new Exception('Erreur de préparation de la requête');
            }
            
            $stmt->bind_param('iss', $is_active, $phase_name, $page_type);
            
            if (!$stmt->execute()) {
                throw new Exception('Erreur d\'exécution de la requête');
            }
            
            $stmt->close();
            echo json_encode(['success' => true, 'message' => 'Phase mise à jour avec succès']);
        }
    } else {
        // La phase n'existe pas, la créer
        $stmt = $mysqli->prepare("INSERT INTO phase_management (phase_name, page_type, is_active) VALUES (?, ?, ?)");
        if (!$stmt) {
            throw new Exception('Erreur de préparation de la requête d\'insertion');
        }
        
        $stmt->bind_param('ssi', $phase_name, $page_type, $is_active);
        
        if (!$stmt->execute()) {
            throw new Exception('Erreur d\'exécution de la requête d\'insertion: ' . $mysqli->error);
        }
        
        $stmt->close();
        echo json_encode(['success' => true, 'message' => 'Phase créée avec succès']);
    }
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Erreur: ' . $e->getMessage()]);
} catch (Error $e) {
    echo json_encode(['success' => false, 'message' => 'Erreur fatale: ' . $e->getMessage()]);
}
?>