<?php
session_start();
require_once '../includes/db.php';

// Vérifier si l'utilisateur est admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../welcome.php');
    exit;
}

// $mysqli est défini dans includes/db.php

// Fonction pour vérifier les questions d'une table
function checkQuestions($mysqli, $table_name) {
    $issues = [];
    $total_questions = 0;
    
    $result = $mysqli->query("SELECT id, question, answer, option1, option2, option3, option4 FROM {$table_name} ORDER BY id");
    
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $total_questions++;
            $answer = trim($row['answer']);
            $options = [
                trim($row['option1']),
                trim($row['option2']),
                trim($row['option3']),
                trim($row['option4'])
            ];
            
            // Normaliser les chaînes pour la comparaison (minuscules, suppression des espaces multiples)
            $normalized_answer = mb_strtolower(preg_replace('/\s+/', ' ', trim($answer)));
            
            // Vérifier si la réponse correspond à l'un des choix (comparaison insensible à la casse)
            $found = false;
            foreach ($options as $index => $option) {
                $normalized_option = mb_strtolower(preg_replace('/\s+/', ' ', trim($option)));
                if ($normalized_answer === $normalized_option) {
                    $found = true;
                    break;
                }
            }
            
            if (!$found) {
                $issues[] = [
                    'id' => $row['id'],
                    'question' => $row['question'],
                    'answer' => $answer,
                    'options' => $options,
                    'normalized_answer' => $normalized_answer,
                    'normalized_options' => array_map(function($opt) {
                        return mb_strtolower(preg_replace('/\s+/', ' ', trim($opt)));
                    }, $options)
                ];
            }
        }
        $result->close();
    }
    
    return [
        'issues' => $issues,
        'total' => $total_questions
    ];
}

// Vérifier les questions françaises
$french_results = checkQuestions($mysqli, 'quizz_questions');
$issues_fr = $french_results['issues'];
$total_fr = $french_results['total'];

// Vérifier les questions arabes
$arabic_results = checkQuestions($mysqli, 'quizz_questions_ar');
$issues_ar = $arabic_results['issues'];
$total_ar = $arabic_results['total'];

$total_questions = $total_fr + $total_ar;
$total_issues = count($issues_fr) + count($issues_ar);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vérification des Questions</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
        .alert-custom {
            border-left: 4px solid #dc3545;
            background-color: #f8d7da;
        }
        .question-item {
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 15px;
            background-color: #fff;
        }
        .question-id {
            font-weight: bold;
            color: #dc3545;
        }
        .answer-missing {
            color: #dc3545;
            font-weight: bold;
        }
        .option-item {
            padding: 5px 10px;
            margin: 5px 0;
            background-color: #f8f9fa;
            border-radius: 4px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1><i class="fas fa-search"></i> Vérification des Questions</h1>
            <a href="questions.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Retour
            </a>
        </div>

        <div class="alert alert-info">
            <i class="fas fa-info-circle"></i> 
            <strong>Total de questions analysées :</strong> <?= $total_questions ?>
            <br>
            <small>
                🇫🇷 Françaises: <?= $total_fr ?> questions 
                <?php if (!empty($issues_fr)): ?>
                    <span class="badge bg-danger"><?= count($issues_fr) ?> problème(s)</span>
                <?php else: ?>
                    <span class="badge bg-success">✓ OK</span>
                <?php endif; ?>
                | 
                🇲🇦 Arabes: <?= $total_ar ?> questions 
                <?php if (!empty($issues_ar)): ?>
                    <span class="badge bg-danger"><?= count($issues_ar) ?> problème(s)</span>
                <?php else: ?>
                    <span class="badge bg-success">✓ OK</span>
                <?php endif; ?>
            </small>
        </div>

        <?php if ($total_issues === 0): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i> 
                <strong>Excellent !</strong> Toutes les questions (françaises et arabes) ont une réponse qui correspond à l'un des choix proposés.
            </div>
        <?php else: ?>
            <div class="alert alert-custom">
                <i class="fas fa-exclamation-triangle"></i> 
                <strong><?= $total_issues ?> question(s) problématique(s) trouvée(s)</strong>
            </div>

            <!-- Onglets pour séparer français et arabe -->
            <ul class="nav nav-tabs mb-4" id="issueTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link <?= !empty($issues_fr) ? 'active' : '' ?>" id="french-issues-tab" data-bs-toggle="tab" data-bs-target="#french-issues" type="button" role="tab">
                        🇫🇷 Questions Françaises 
                        <?php if (!empty($issues_fr)): ?>
                            <span class="badge bg-danger"><?= count($issues_fr) ?></span>
                        <?php endif; ?>
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link <?= empty($issues_fr) && !empty($issues_ar) ? 'active' : '' ?>" id="arabic-issues-tab" data-bs-toggle="tab" data-bs-target="#arabic-issues" type="button" role="tab">
                        🇲🇦 Questions Arabes 
                        <?php if (!empty($issues_ar)): ?>
                            <span class="badge bg-danger"><?= count($issues_ar) ?></span>
                        <?php endif; ?>
                    </button>
                </li>
            </ul>

            <div class="tab-content" id="issueTabsContent">
                <!-- Onglet Questions Françaises -->
                <div class="tab-pane fade <?= !empty($issues_fr) ? 'show active' : '' ?>" id="french-issues" role="tabpanel">
                    <?php if (empty($issues_fr)): ?>
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle"></i> 
                            Toutes les questions françaises sont correctes !
                        </div>
                    <?php else: ?>
                        <div class="mt-4">
                            <?php foreach ($issues_fr as $issue): ?>
                                <div class="question-item">
                                    <div class="mb-2">
                                        <span class="question-id">Question #<?= htmlspecialchars($issue['id']) ?></span>
                                    </div>
                                    <div class="mb-3">
                                        <strong>Question :</strong> <?= htmlspecialchars($issue['question']) ?>
                                    </div>
                                    <div class="mb-2">
                                        <span class="answer-missing">
                                            <i class="fas fa-times-circle"></i> 
                                            <strong>Réponse attendue :</strong> "<?= htmlspecialchars($issue['answer']) ?>"
                                        </span>
                                        <?php if (isset($issue['normalized_answer'])): ?>
                                            <br><small class="text-muted">(Normalisée: "<?= htmlspecialchars($issue['normalized_answer']) ?>")</small>
                                        <?php endif; ?>
                                    </div>
                                    <div class="mt-3">
                                        <strong>Choix disponibles :</strong>
                                        <ul class="list-unstyled mt-2">
                                            <?php foreach ($issue['options'] as $index => $option): ?>
                                                <li class="option-item">
                                                    Option <?= $index + 1 ?>: "<?= htmlspecialchars($option) ?>"
                                                    <?php if (isset($issue['normalized_options'][$index])): ?>
                                                        <br><small class="text-muted">(Normalisée: "<?= htmlspecialchars($issue['normalized_options'][$index]) ?>")</small>
                                                    <?php endif; ?>
                                                </li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Onglet Questions Arabes -->
                <div class="tab-pane fade <?= empty($issues_fr) && !empty($issues_ar) ? 'show active' : '' ?>" id="arabic-issues" role="tabpanel">
                    <?php if (empty($issues_ar)): ?>
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle"></i> 
                            جميع الأسئلة العربية صحيحة!
                        </div>
                    <?php else: ?>
                        <div class="mt-4">
                            <?php foreach ($issues_ar as $issue): ?>
                                <div class="question-item">
                                    <div class="mb-2">
                                        <span class="question-id">السؤال #<?= htmlspecialchars($issue['id']) ?></span>
                                    </div>
                                    <div class="mb-3" style="direction: rtl; text-align: right;">
                                        <strong>السؤال :</strong> <?= htmlspecialchars($issue['question']) ?>
                                    </div>
                                    <div class="mb-2">
                                        <span class="answer-missing">
                                            <i class="fas fa-times-circle"></i> 
                                            <strong>الإجابة المتوقعة :</strong> "<?= htmlspecialchars($issue['answer']) ?>"
                                        </span>
                                        <?php if (isset($issue['normalized_answer'])): ?>
                                            <br><small class="text-muted">(معيارية: "<?= htmlspecialchars($issue['normalized_answer']) ?>")</small>
                                        <?php endif; ?>
                                    </div>
                                    <div class="mt-3" style="direction: rtl; text-align: right;">
                                        <strong>الخيارات المتاحة :</strong>
                                        <ul class="list-unstyled mt-2">
                                            <?php foreach ($issue['options'] as $index => $option): ?>
                                                <li class="option-item">
                                                    الخيار <?= $index + 1 ?>: "<?= htmlspecialchars($option) ?>"
                                                    <?php if (isset($issue['normalized_options'][$index])): ?>
                                                        <br><small class="text-muted">(معيارية: "<?= htmlspecialchars($issue['normalized_options'][$index]) ?>")</small>
                                                    <?php endif; ?>
                                                </li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
