<?php
include 'admin_header.php';
require_once '../includes/db.php';

// Fonction pour récupérer toutes les phases
function getAllPhases($mysqli) {
    $phases = [];
    $result = $mysqli->query("SELECT * FROM phase_management ORDER BY page_type, 
        CASE phase_name 
            WHEN 'Phase de groupes' THEN 1
            WHEN '8e de finale' THEN 2
            WHEN 'Quart de finale' THEN 3
            WHEN 'Demi-finale' THEN 4
            WHEN 'Finale' THEN 5
            ELSE 6
        END");
    while ($row = $result->fetch_assoc()) {
        $phases[] = $row;
    }
    return $phases;
}

// Fonction pour mettre à jour l'état d'une phase
function updatePhaseStatus($mysqli, $phase_name, $page_type, $is_active) {
    try {
        // Mettre à jour la phase
        $stmt = $mysqli->prepare("UPDATE phase_management SET is_active = ? WHERE phase_name = ? AND page_type = ?");
        if (!$stmt) {
            throw new Exception('Erreur de préparation de la requête de mise à jour');
        }
        
        $stmt->bind_param('iss', $is_active, $phase_name, $page_type);
        if (!$stmt->execute()) {
            throw new Exception('Erreur d\'exécution de la requête de mise à jour');
        }
        
        $affected_rows = $stmt->affected_rows;
        $stmt->close();
        
        return $affected_rows > 0;
        
    } catch (Exception $e) {
        error_log("Erreur updatePhaseStatus: " . $e->getMessage());
        return false;
    }
}

// Le traitement AJAX est maintenant dans ajax_phase_management.php

$phases = getAllPhases($mysqli);
?>

<style>
.phase-management-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 20px;
}

.phases-table-container {
    background: #2a2a3a;
    border-radius: 12px;
    padding: 20px;
    border: 2px solid #ffb30044;
    margin-top: 20px;
}

.phases-table {
    margin-bottom: 0;
    background: transparent;
}

.phases-table thead th {
    background: #1e293b;
    color: #ffb300;
    font-weight: 700;
    border: none;
    padding: 15px 10px;
    text-align: center;
    vertical-align: middle;
}

.phases-table tbody td {
    border: none;
    padding: 12px 10px;
    vertical-align: middle;
    border-bottom: 1px solid #374151;
}

.phase-row:hover {
    background: rgba(255, 179, 0, 0.05);
}

.phase-row.phase-groups {
    background: rgba(255, 193, 7, 0.1);
    border-left: 4px solid #ffc107;
}

.phase-row.phase-groups .phase-name-cell {
    color: #ffc107;
    font-weight: 700;
}

.phase-row.phase-groups .toggle-switch-small input:not(:checked) {
    opacity: 0.5;
    cursor: not-allowed;
}

.phase-row.phase-groups .action-buttons .btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.phase-name-cell {
    color: #fff;
    font-size: 1rem;
    font-weight: 600;
}

.status-cell {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
}

.status-indicator {
    width: 10px;
    height: 10px;
    border-radius: 50%;
    display: inline-block;
}

.status-indicator.active {
    background: #4caf50;
    box-shadow: 0 0 8px rgba(76, 175, 80, 0.5);
}

.status-indicator.inactive {
    background: #f44336;
    box-shadow: 0 0 8px rgba(244, 67, 54, 0.5);
}

.toggle-switch-small {
    position: relative;
    display: inline-block;
    width: 40px;
    height: 20px;
}

.toggle-switch-small input {
    opacity: 0;
    width: 0;
    height: 0;
}

.slider-small {
    position: absolute;
    cursor: pointer;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background-color: #f44336;
    transition: 0.3s;
    border-radius: 20px;
}

.slider-small:before {
    position: absolute;
    content: "";
    height: 16px;
    width: 16px;
    left: 2px;
    bottom: 2px;
    background-color: white;
    transition: 0.3s;
    border-radius: 50%;
}

input:checked + .slider-small {
    background-color: #4caf50;
}

input:checked + .slider-small:before {
    transform: translateX(20px);
}

.slider-small:hover {
    box-shadow: 0 0 8px rgba(255, 179, 0, 0.3);
}

.action-buttons {
    display: flex;
    gap: 8px;
    justify-content: center;
    flex-wrap: wrap;
}

.action-buttons .btn {
    font-size: 0.8rem;
    padding: 4px 8px;
    border-radius: 6px;
    transition: all 0.3s ease;
}

.action-buttons .btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.btn-success {
    background: #4caf50;
    border-color: #4caf50;
}

.btn-warning {
    background: #ff9800;
    border-color: #ff9800;
    color: #fff;
}

.btn-info {
    background: #17a2b8;
    border-color: #17a2b8;
    color: #fff;
}

.btn-outline-secondary {
    border-color: #6c757d;
    color: #6c757d;
}

.btn-outline-secondary:hover {
    background: #6c757d;
    border-color: #6c757d;
    color: #fff;
}

@media (max-width: 768px) {
    .phase-management-container {
        padding: 10px;
    }
    
    .phases-table-container {
        padding: 15px;
    }
    
    .phases-table thead th {
        padding: 10px 5px;
        font-size: 0.9rem;
    }
    
    .phases-table tbody td {
        padding: 8px 5px;
        font-size: 0.85rem;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 4px;
    }
    
    .action-buttons .btn {
        font-size: 0.75rem;
        padding: 3px 6px;
    }
    
    .toggle-switch-small {
        width: 35px;
        height: 18px;
    }
    
    .slider-small:before {
        height: 14px;
        width: 14px;
    }
    
    input:checked + .slider-small:before {
        transform: translateX(17px);
    }
}
</style>

<div class="phase-management-container">
    <h2 class="text-center mb-4" style="color: #ffb300; font-weight: 800;">
        <i class="fas fa-cogs"></i> Gestion des Phases
    </h2>
    
    <div class="alert alert-info" style="background: #1e3a5f; color: #fff; border: 2px solid #2196f3; border-radius: 10px; padding: 15px; margin-bottom: 30px;">
        <i class="fas fa-info-circle"></i>
        <strong>Instructions :</strong> Utilisez les interrupteurs ci-dessous pour activer ou désactiver les onglets des phases dans les pages Quiz, Gagnants et Défis Photos.
    </div>

    <?php
    // Grouper les phases par type de page
    $quiz_phases = [];
    $gagnants_phases = [];
    $challenge_photos_phases = [];
    
    foreach ($phases as $phase) {
        if ($phase['page_type'] === 'quiz') {
            $quiz_phases[] = $phase;
        } elseif ($phase['page_type'] === 'gagnants') {
            $gagnants_phases[] = $phase;
        } elseif ($phase['page_type'] === 'challenge_photos') {
            $challenge_photos_phases[] = $phase;
        }
    }
    ?>

    <!-- Tableau compact avec phases chronologiques -->
    <div class="phases-table-container">
        <div class="table-responsive">
            <table class="table table-dark phases-table">
                <thead>
                    <tr>
                        <th style="width: 20%;">Phase</th>
                        <th style="width: 15%;" class="text-center">Quiz</th>
                        <th style="width: 15%;" class="text-center">Gagnants</th>
                        <th style="width: 15%;" class="text-center">Défis Photos</th>
                        <th style="width: 35%;" class="text-center">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    // Ordre chronologique des phases
                    $chronological_phases = [
                        'Phase de groupes',
                        '8e de finale', 
                        'Quart de finale',
                        'Demi-finale',
                        'Finale'
                    ];
                    
                    foreach ($chronological_phases as $phase_name):
                        // Récupérer l'état pour Quiz
                        $quiz_phase = null;
                        foreach ($quiz_phases as $phase) {
                            if ($phase['phase_name'] === $phase_name) {
                                $quiz_phase = $phase;
                                break;
                            }
                        }
                        
                        // Récupérer l'état pour Gagnants
                        $gagnants_phase = null;
                        foreach ($gagnants_phases as $phase) {
                            if ($phase['phase_name'] === $phase_name) {
                                $gagnants_phase = $phase;
                                break;
                            }
                        }
                        
                        // Récupérer l'état pour Défis Photos
                        $challenge_photos_phase = null;
                        foreach ($challenge_photos_phases as $phase) {
                            if ($phase['phase_name'] === $phase_name) {
                                $challenge_photos_phase = $phase;
                                break;
                            }
                        }
                    ?>
                    <tr class="phase-row <?= $phase_name === 'Phase de groupes' ? 'phase-groups' : '' ?>">
                        <td>
                            <div class="phase-name-cell">
                                <strong><?= htmlspecialchars($phase_name) ?></strong>
                            </div>
                        </td>
                        
                        <!-- Quiz Status -->
                        <td class="text-center">
                            <?php if ($quiz_phase): ?>
                                <div class="status-cell">
                                    <span class="status-indicator <?= $quiz_phase['is_active'] ? 'active' : 'inactive' ?>"></span>
                                    <label class="toggle-switch-small">
                                        <input type="checkbox" 
                                               <?= $quiz_phase['is_active'] ? 'checked' : '' ?>
                                               onchange="togglePhase('<?= htmlspecialchars($phase_name, ENT_QUOTES) ?>', 'quiz', this.checked)">
                                        <span class="slider-small"></span>
                                    </label>
                                </div>
                            <?php else: ?>
                                <!-- Afficher une checkbox même si la phase n'existe pas encore (sera créée automatiquement) -->
                                <div class="status-cell">
                                    <span class="status-indicator inactive"></span>
                                    <label class="toggle-switch-small">
                                        <input type="checkbox" 
                                               onchange="togglePhase('<?= htmlspecialchars($phase_name, ENT_QUOTES) ?>', 'quiz', this.checked)">
                                        <span class="slider-small"></span>
                                    </label>
                                </div>
                            <?php endif; ?>
                        </td>
                        
                        <!-- Gagnants Status -->
                        <td class="text-center">
                            <?php if ($gagnants_phase): ?>
                                <div class="status-cell">
                                    <span class="status-indicator <?= $gagnants_phase['is_active'] ? 'active' : 'inactive' ?>"></span>
                                    <label class="toggle-switch-small">
                                        <input type="checkbox" 
                                               <?= $gagnants_phase['is_active'] ? 'checked' : '' ?>
                                               onchange="togglePhase('<?= htmlspecialchars($phase_name, ENT_QUOTES) ?>', 'gagnants', this.checked)">
                                        <span class="slider-small"></span>
                                    </label>
                                </div>
                            <?php else: ?>
                                <span class="text-muted">-</span>
                            <?php endif; ?>
                        </td>
                        
                        <!-- Défis Photos Status -->
                        <td class="text-center">
                            <?php if ($challenge_photos_phase): ?>
                                <div class="status-cell">
                                    <span class="status-indicator <?= $challenge_photos_phase['is_active'] ? 'active' : 'inactive' ?>"></span>
                                    <label class="toggle-switch-small">
                                        <input type="checkbox" 
                                               <?= $challenge_photos_phase['is_active'] ? 'checked' : '' ?>
                                               onchange="togglePhase('<?= htmlspecialchars($phase_name, ENT_QUOTES) ?>', 'challenge_photos', this.checked)">
                                        <span class="slider-small"></span>
                                    </label>
                                </div>
                            <?php else: ?>
                                <span class="text-muted">-</span>
                            <?php endif; ?>
                        </td>
                        
                        <!-- Actions -->
                        <td class="text-center">
                            <div class="action-buttons">
                                <?php if ($quiz_phase): ?>
                                    <button class="btn btn-sm <?= $quiz_phase['is_active'] ? 'btn-success' : 'btn-outline-secondary' ?>" 
                                            onclick="togglePhase('<?= htmlspecialchars($phase_name, ENT_QUOTES) ?>', 'quiz', <?= $quiz_phase['is_active'] ? 'false' : 'true' ?>)">
                                        <i class="fas fa-question-circle"></i> Quiz
                                    </button>
                                <?php else: ?>
                                    <!-- Bouton pour créer et activer la phase si elle n'existe pas encore -->
                                    <button class="btn btn-sm btn-outline-secondary" 
                                            onclick="togglePhase('<?= htmlspecialchars($phase_name, ENT_QUOTES) ?>', 'quiz', true)"
                                            title="Créer et activer cette phase">
                                        <i class="fas fa-question-circle"></i> Quiz
                                    </button>
                                <?php endif; ?>
                                
                                <?php if ($gagnants_phase): ?>
                                    <button class="btn btn-sm <?= $gagnants_phase['is_active'] ? 'btn-warning' : 'btn-outline-secondary' ?>" 
                                            onclick="togglePhase('<?= htmlspecialchars($phase_name, ENT_QUOTES) ?>', 'gagnants', <?= $gagnants_phase['is_active'] ? 'false' : 'true' ?>)">
                                        <i class="fas fa-trophy"></i> Gagnants
                                    </button>
                                <?php endif; ?>
                                
                                <?php if ($challenge_photos_phase): ?>
                                    <button class="btn btn-sm <?= $challenge_photos_phase['is_active'] ? 'btn-info' : 'btn-outline-secondary' ?>" 
                                            onclick="togglePhase('<?= htmlspecialchars($phase_name, ENT_QUOTES) ?>', 'challenge_photos', <?= $challenge_photos_phase['is_active'] ? 'false' : 'true' ?>)">
                                        <i class="fas fa-camera"></i> Photos
                                    </button>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
function togglePhase(phaseName, pageType, isActive) {
    console.log('togglePhase appelé:', {phaseName, pageType, isActive});
    
    // Désactiver tous les contrôles pendant la requête
    const allControls = document.querySelectorAll('button, input[type="checkbox"]');
    allControls.forEach(control => {
        control.disabled = true;
        control.style.opacity = '0.5';
    });
    
    // Préparer les données
    const formData = new FormData();
    formData.append('action', 'toggle_phase');
    formData.append('phase_name', phaseName);
    formData.append('page_type', pageType);
    formData.append('is_active', isActive ? 1 : 0);
    
    console.log('Envoi de la requête...');
    
    // Envoyer la requête
    fetch('ajax_phase_management.php', {
        method: 'POST',
        body: formData
    })
    .then(response => {
        console.log('Réponse reçue:', response.status);
        
        if (!response.ok) {
            throw new Error(`Erreur HTTP ${response.status}: ${response.statusText}`);
        }
        
        return response.text();
    })
    .then(text => {
        console.log('Texte reçu:', text);
        
        try {
            const data = JSON.parse(text);
            console.log('Données parsées:', data);
            
            if (data.success) {
                showNotification(data.message || 'Phase mise à jour avec succès', 'success');
                // Actualiser la page après un court délai
                setTimeout(() => {
                    location.reload();
                }, 1500);
            } else {
                showNotification(data.message || 'Erreur lors de la mise à jour', 'error');
                // Réactiver les contrôles en cas d'erreur
                allControls.forEach(control => {
                    control.disabled = false;
                    control.style.opacity = '1';
                });
            }
        } catch (e) {
            console.error('Erreur de parsing JSON:', e);
            console.error('Texte reçu:', text);
            showNotification('Erreur de format de réponse du serveur', 'error');
            // Réactiver les contrôles en cas d'erreur
            allControls.forEach(control => {
                control.disabled = false;
                control.style.opacity = '1';
            });
        }
    })
    .catch(error => {
        console.error('Erreur de requête:', error);
        showNotification('Erreur de connexion: ' + error.message, 'error');
        // Réactiver les contrôles en cas d'erreur
        allControls.forEach(control => {
            control.disabled = false;
            control.style.opacity = '1';
        });
    });
}

function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 8px;
        color: white;
        font-weight: 600;
        z-index: 9999;
        animation: slideIn 0.3s ease;
        ${type === 'success' ? 'background: #4caf50;' : 'background: #f44336;'}
    `;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, 3000);
}

// Ajouter les styles d'animation
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);
</script>

<?php include 'admin_footer.php'; ?>
