<?php
header('Content-Type: application/json');
require_once '../includes/db.php';
require_once '../includes/email.php';

$email = isset($_POST['email']) ? trim($_POST['email']) : '';

if (empty($email)) {
    echo json_encode(['success' => false, 'message' => 'L\'email est obligatoire.']);
    exit;
}

// Validation du format email
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    echo json_encode(['success' => false, 'message' => 'Format d\'email invalide.']);
    exit;
}

try {
    // Vérifier si l'email existe dans la base de données
    $stmt = $mysqli->prepare('SELECT id, username FROM Users WHERE email = ?');
    
    if (!$stmt) {
        throw new Exception('Erreur de préparation de la requête: ' . $mysqli->error);
    }
    
    $stmt->bind_param('s', $email);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();

    if (!$user) {
        // Pour des raisons de sécurité, on ne révèle pas si l'email existe ou non
        echo json_encode([
            'success' => true, 
            'message' => 'Si cet email existe dans notre système, vous recevrez un lien de réinitialisation.'
        ]);
        exit;
    }

    // Générer un token de réinitialisation sécurisé (64 caractères hexadécimaux)
    $token = bin2hex(random_bytes(32));
    
    // Vérifier que le token a bien 64 caractères
    if (strlen($token) !== 64) {
        throw new Exception('Erreur lors de la génération du token.');
    }
    
    // Token valide 1 heure
    $expires = date('Y-m-d H:i:s', strtotime('+1 hour'));
    
    // Supprimer les anciens tokens pour cet utilisateur (nettoyage)
    $deleteStmt = $mysqli->prepare('DELETE FROM password_reset_tokens WHERE user_id = ?');
    
    if (!$deleteStmt) {
        throw new Exception('Erreur de préparation de la requête de suppression: ' . $mysqli->error);
    }
    
    $deleteStmt->bind_param('i', $user['id']);
    $deleteStmt->execute();
    $deleteStmt->close();

    // Insérer le nouveau token
    $insertStmt = $mysqli->prepare('INSERT INTO password_reset_tokens (user_id, token, expires_at) VALUES (?, ?, ?)');
    
    if (!$insertStmt) {
        throw new Exception('Erreur de préparation de la requête d\'insertion: ' . $mysqli->error);
    }
    
    $insertStmt->bind_param('iss', $user['id'], $token, $expires);
    
    if (!$insertStmt->execute()) {
        $insertStmt->close();
        throw new Exception('Erreur lors de l\'insertion du token: ' . $mysqli->error);
    }
    
    $insertStmt->close();
    
    // Construire l'URL de réinitialisation
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $basePath = dirname(dirname($_SERVER['SCRIPT_NAME']));
    $baseUrl = $protocol . '://' . $host . $basePath;
    $resetUrl = $baseUrl . '/reset_password.php?token=' . $token;
    
    // Envoyer l'email de réinitialisation
    $emailSent = sendPasswordResetEmail($email, $user['username'], $resetUrl);
    
    if ($emailSent) {
        echo json_encode([
            'success' => true, 
            'message' => 'Un lien de réinitialisation a été envoyé à votre adresse email. Veuillez vérifier votre boîte de réception (et vos spams).'
        ]);
    } else {
        // Même si l'email n'a pas pu être envoyé, on retourne un succès pour la sécurité
        // Mais on log l'erreur
        error_log('Failed to send password reset email to: ' . $email);
        echo json_encode([
            'success' => true, 
            'message' => 'Si cet email existe dans notre système, vous recevrez un lien de réinitialisation.'
        ]);
    }
    
} catch (Exception $e) {
    error_log('Forgot password error: ' . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'Erreur serveur. Veuillez réessayer plus tard.'
    ]);
}
?>
