<?php
date_default_timezone_set('Etc/GMT-1');
header('Content-Type: application/json');

// Inclure le système de langue en premier
require_once '../includes/language.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../includes/db.php';

// Traductions des noms d'équipes en arabe
$equipes_ar = [
    'Maroc' => 'المغرب', 'Comores' => 'جزر القمر', 'Mali' => 'مالي', 'Zambie' => 'زامبيا',
    'Afrique du Sud' => 'جنوب أفريقيا', 'Angola' => 'أنغولا', 'Égypte' => 'مصر', 'Zimbabwe' => 'زيمبابوي',
    'Sénégal' => 'السنغال', 'Botswana' => 'بوتسوانا', 'Tunisie' => 'تونس', 'Ouganda' => 'أوغندا',
    'RD Congo' => 'جمهورية الكونغو الديمقراطية', 'Bénin' => 'بنين', 'Nigeria' => 'نيجيريا', 'Tanzanie' => 'تنزانيا',
    'Algérie' => 'الجزائر', 'Soudan' => 'السودان', 'Côte d\'Ivoire' => 'ساحل العاج', 'Mozambique' => 'موزمبيق',
    'Cameroun' => 'الكاميرون', 'Gabon' => 'الغابون', 'Burkina Faso' => 'بوركينا فاسو', 'Guinée Équatoriale' => 'غينيا الاستوائية'
];

// Fonction pour obtenir le nom de l'équipe selon la langue
function getTeamName($team_name, $current_lang) {
    global $equipes_ar;
    if ($current_lang === 'ar') {
        return $equipes_ar[$team_name] ?? $team_name;
    } else {
        return $team_name;
    }
}

function getFlagUrl($team) {
    $map = [
        'Maroc' => 'ma',
        'Comores' => 'km',
        'Mali' => 'ml',
        'Zambie' => 'zm',
        'Afrique du Sud' => 'za',
        'Angola' => 'ao',
        'Égypte' => 'eg',
        'Zimbabwe' => 'zw',
        'Sénégal' => 'sn',
        'Botswana' => 'bw',
        'Tunisie' => 'tn',
        'Ouganda' => 'ug',
        'RD Congo' => 'cd',
        'Bénin' => 'bj',
        'Nigeria' => 'ng',
        'Tanzanie' => 'tz',
        'Algérie' => 'dz',
        'Soudan' => 'sd',
        'Côte d\'Ivoire' => 'ci',
        'Mozambique' => 'mz',
        'Cameroun' => 'cm',
        'Gabon' => 'ga',
        'Burkina Faso' => 'bf',
        'Guinée Équatoriale' => 'gq',
    ];
    $code = $map[$team] ?? 'un';
    return 'https://flagcdn.com/w80/' . $code . '.png';
}

if(!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => __('unauthorized')]);
    exit;
}
$user_id = $_SESSION['user_id'];
$current_lang = getCurrentLanguage();

// Fonction pour obtenir le titre traduit d'une phase
function getPhaseTitre($phase, $current_lang) {
    $phase_translations = [
        'fr' => [
            'Phase de groupes' => 'Phase de groupes',
            '8e de finale' => '8e de finale',
            'Quart de finale' => 'Quart de finale',
            'Demi-finale' => 'Demi-finale',
            'Match pour la 3e place' => 'Match pour la 3e place',
            'Finale' => 'Finale'
        ],
        'ar' => [
            'Phase de groupes' => 'مرحلة المجموعات',
            '8e de finale' => 'دور الثمن النهائي',
            'Quart de finale' => 'ربع النهائي',
            'Demi-finale' => 'نصف النهائي',
            'Match pour la 3e place' => 'مباراة المركز الثالث',
            'Match 3e place' => 'مباراة المركز الثالث',
            'Finale' => 'النهائي'
        ]
    ];
    
    return $phase_translations[$current_lang][$phase] ?? $phase;
}

// Fonction pour obtenir le titre traduit d'une journée
function getJourneeTitre($journee, $current_lang) {
    // Les journées arrivent en français car les traductions dans lang/ar.php ne sont pas appliquées
    if ($current_lang === 'ar') {
        // Rechercher le pattern français "Phase de groupes · Journée X sur Y"
        if (preg_match('/Phase de groupes · Journée (\d+) sur (\d+)/', $journee, $matches)) {
            $numero = $matches[1];
            $total = $matches[2];
            // Convertir le numéro en ordinal arabe
            $ordinaux = ['', 'الأولى', 'الثانية', 'الثالثة', 'الرابعة', 'الخامسة'];
            $ordinal = $ordinaux[$numero] ?? $numero;
            return "الجولة {$ordinal} من {$total}";
        }
        // Rechercher le pattern arabe "مرحلة المجموعات · الجولة X من Y"
        if (preg_match('/مرحلة المجموعات · الجولة (\d+) من (\d+)/', $journee, $matches)) {
            $numero = $matches[1];
            $total = $matches[2];
            $ordinaux = ['', 'الأولى', 'الثانية', 'الثالثة', 'الرابعة', 'الخامسة'];
            $ordinal = $ordinaux[$numero] ?? $numero;
            return "الجولة {$ordinal} من {$total}";
        }
        // Rechercher aussi le pattern "Journée X sur Y" seul
        if (preg_match('/Journée (\d+) sur (\d+)/', $journee, $matches)) {
            $numero = $matches[1];
            $total = $matches[2];
            $ordinaux = ['', 'الأولى', 'الثانية', 'الثالثة', 'الرابعة', 'الخامسة'];
            $ordinal = $ordinaux[$numero] ?? $numero;
            return "الجولة {$ordinal} من {$total}";
        }
        // Rechercher aussi le pattern "الجولة X من Y" seul
        if (preg_match('/الجولة (\d+) من (\d+)/', $journee, $matches)) {
            $numero = $matches[1];
            $total = $matches[2];
            $ordinaux = ['', 'الأولى', 'الثانية', 'الثالثة', 'الرابعة', 'الخامسة'];
            $ordinal = $ordinaux[$numero] ?? $numero;
            return "الجولة {$ordinal} من {$total}";
        }
        // Rechercher aussi le pattern "Journée X sur Y" avec espaces
        if (preg_match('/Journée\s+(\d+)\s+sur\s+(\d+)/', $journee, $matches)) {
            $numero = $matches[1];
            $total = $matches[2];
            $ordinaux = ['', 'الأولى', 'الثانية', 'الثالثة', 'الرابعة', 'الخامسة'];
            $ordinal = $ordinaux[$numero] ?? $numero;
            return "الجولة {$ordinal} من {$total}";
        }
        // Rechercher aussi le pattern "Journée X sur Y" avec espaces variables
        if (preg_match('/Journée\s*(\d+)\s+sur\s*(\d+)/', $journee, $matches)) {
            $numero = $matches[1];
            $total = $matches[2];
            $ordinaux = ['', 'الأولى', 'الثانية', 'الثالثة', 'الرابعة', 'الخامسة'];
            $ordinal = $ordinaux[$numero] ?? $numero;
            return "الجولة {$ordinal} من {$total}";
        }
        // Regex plus générique pour capturer tous les cas
        if (preg_match('/(\d+)\s+sur\s+(\d+)/', $journee, $matches)) {
            $numero = $matches[1];
            $total = $matches[2];
            $ordinaux = ['', 'الأولى', 'الثانية', 'الثالثة', 'الرابعة', 'الخامسة'];
            $ordinal = $ordinaux[$numero] ?? $numero;
            return "الجولة {$ordinal} من {$total}";
        }
    } else {
        // Pour le français, extraire la partie "Journée X sur Y"
        if (preg_match('/Phase de groupes · Journée (\d+) sur (\d+)/', $journee, $matches)) {
            return "Journée {$matches[1]} sur {$matches[2]}";
        }
        if (preg_match('/Journée (\d+) sur (\d+)/', $journee, $matches)) {
            return "Journée {$matches[1]} sur {$matches[2]}";
        }
    }
    return $journee;
}

// Lire la phase à afficher (définie par l'admin)
$phase_file = __DIR__.'/../includes/phase_dashboard.txt';
$phase_to_show = file_exists($phase_file) ? trim(file_get_contents($phase_file)) : '';

// Récupérer tous les matchs de l'utilisateur, toutes phases confondues
$stmt = $mysqli->prepare('SELECT m.*, p.predicted_team_1, p.predicted_team_2, p.id AS prediction_id FROM Matches m LEFT JOIN Predictions p ON p.match_id = m.id AND p.user_id = ? ORDER BY m.phase, m.match_date ASC');
$stmt->bind_param('i', $user_id);
$stmt->execute();
$result = $stmt->get_result();
$matches = [];
while ($row = $result->fetch_assoc()) {
    $matches[] = $row;
}
$stmt->close();

$grouped = [];
foreach($matches as $m) {
    $phase = $m['phase'] ?: __('group_phase');
    $grouped[$phase][] = $m;
}

// Phases de groupes en français (comme dans la base de données)
$phases_groupes_fr = [
    'Phase de groupes · Journée 1 sur 3',
    'Phase de groupes · Journée 2 sur 3',
    'Phase de groupes · Journée 3 sur 3'
];

// Phases de groupes traduites (pour l'affichage)
$phases_groupes = [
    __('group_phase') . " · " . __('day') . " 1 " . __('of') . " 3",
    __('group_phase') . " · " . __('day') . " 2 " . __('of') . " 3",
    __('group_phase') . " · " . __('day') . " 3 " . __('of') . " 3"
];

$phases_officielles = [
    __('round_of_16'),
    __('quarter_final'),
    __('semi_final'),
    __('third_place_match'),
    __('final')
];

$groupes_matches = [];
$other_phases = [];
foreach($grouped as $phase => $phaseMatches) {
    // Vérifier si c'est une phase de groupes (en français ou traduite)
    $is_group_phase = in_array($phase, $phases_groupes) || in_array($phase, $phases_groupes_fr);
    
    // Vérifier aussi si la phase contient "Phase de groupes" ou "Journée" pour capturer toutes les variantes
    if (!$is_group_phase) {
        $phase_lower = strtolower($phase);
        $is_group_phase = (stripos($phase_lower, 'phase de groupes') !== false || 
                          stripos($phase_lower, 'journée') !== false ||
                          stripos($phase_lower, 'مرحلة المجموعات') !== false ||
                          stripos($phase_lower, 'الجولة') !== false);
    }
    
    if ($is_group_phase) {
        // Normaliser la phase pour utiliser la version traduite
        if (in_array($phase, $phases_groupes_fr)) {
            // Trouver l'index correspondant
            $index = array_search($phase, $phases_groupes_fr);
            $normalized_phase = $phases_groupes[$index] ?? $phase;
            $groupes_matches[$normalized_phase] = $phaseMatches;
        } else {
            // Si la phase ne correspond pas exactement, essayer de la mapper
            $mapped = false;
            foreach($phases_groupes_fr as $idx => $phase_fr) {
                if (stripos($phase, 'journée') !== false || stripos($phase, 'الجولة') !== false) {
                    // Extraire le numéro de journée
                    preg_match('/(\d+)/', $phase, $matches);
                    if (!empty($matches[1])) {
                        $journee_num = intval($matches[1]);
                        if ($journee_num >= 1 && $journee_num <= 3) {
                            $normalized_phase = $phases_groupes[$journee_num - 1] ?? $phase;
                            if (!isset($groupes_matches[$normalized_phase])) {
                                $groupes_matches[$normalized_phase] = [];
                            }
                            $groupes_matches[$normalized_phase] = array_merge($groupes_matches[$normalized_phase], $phaseMatches);
                            $mapped = true;
                            break;
                        }
                    }
                }
            }
            if (!$mapped) {
                $groupes_matches[$phase] = $phaseMatches;
            }
        }
    } else {
        // Pour les autres phases, vérifier si elles correspondent aux phases officielles ou les ajouter quand même
        $phase_normalized = null;
        
        // Mapping des phases françaises vers les phases traduites
        $phase_fr_to_officielle = [
            '8e de finale' => __('round_of_16'),
            'Quart de finale' => __('quarter_final'),
            'Demi-finale' => __('semi_final'),
            'Match pour la 3e place' => __('third_place_match'),
            'Finale' => __('final'),
            // Variantes possibles
            'finale' => __('final'),
            'Final' => __('final'),
            'final' => __('final')
        ];
        
        // Vérifier d'abord si la phase correspond exactement à une phase officielle
        if (in_array($phase, $phases_officielles)) {
            $phase_normalized = $phase;
        }
        // Sinon, vérifier si c'est une phase française qui doit être mappée
        else if (isset($phase_fr_to_officielle[$phase])) {
            $phase_normalized = $phase_fr_to_officielle[$phase];
        }
        // Sinon, essayer de trouver une correspondance par comparaison
        else {
            foreach($phases_officielles as $phase_officielle) {
                $phase_fr_mapping = [
                    __('round_of_16') => '8e de finale',
                    __('quarter_final') => 'Quart de finale',
                    __('semi_final') => 'Demi-finale',
                    __('third_place_match') => 'Match pour la 3e place',
                    __('final') => 'Finale'
                ];
                $phase_fr = $phase_fr_mapping[$phase_officielle] ?? $phase_officielle;
                
                // Comparer la phase avec les variantes possibles (insensible à la casse)
                $phase_lower = strtolower(trim($phase));
                $phase_fr_lower = strtolower(trim($phase_fr));
                $phase_officielle_lower = strtolower(trim($phase_officielle));
                
                if ($phase_lower === $phase_officielle_lower || 
                    $phase_lower === $phase_fr_lower ||
                    stripos($phase, $phase_fr) !== false ||
                    stripos($phase_fr, $phase) !== false ||
                    stripos($phase, $phase_officielle) !== false ||
                    stripos($phase_officielle, $phase) !== false) {
                    $phase_normalized = $phase_officielle;
                    break;
                }
            }
        }
        
        // Si aucune correspondance trouvée, utiliser la phase telle quelle
        if ($phase_normalized === null) {
            $phase_normalized = $phase;
        }
        
        if (!isset($other_phases[$phase_normalized])) {
            $other_phases[$phase_normalized] = [];
        }
        $other_phases[$phase_normalized] = array_merge($other_phases[$phase_normalized], $phaseMatches);
    }
}
$html = '';
// Bloc principal pour les groupes (avec sous-tabs)
// Toujours créer le bloc de phase de groupes, même s'il n'y a pas de matchs
$html .= '<div class="phase-block" data-phase="' . __('group_phase') . '">';
if (!empty($groupes_matches)) {
    foreach($phases_groupes as $journee) {
        if (!isset($groupes_matches[$journee])) continue;
        // Extraire seulement la partie journée pour l'attribut data-journee
        $journee_simple = getJourneeTitre($journee, $current_lang);
        $html .= '<div class="journee-block" data-journee="'.htmlspecialchars($journee_simple ?? '').'">';
        $html .= '<h4 class="mb-2" style="color: #ffb300; visibility: hidden;">'.htmlspecialchars($journee ?? '').'</h4>';
        foreach($groupes_matches[$journee] as $m) {
            $team1Flag = getFlagUrl($m['team_1']);
            $team2Flag = getFlagUrl($m['team_2']);
            $date = date('d/m/Y H:i', strtotime($m['match_date'])) ?: '';
            
            // Charger la configuration du délai de clôture des pronostics de matchs
            $match_deadline_hours_file = __DIR__.'/../includes/match_deadline_hours.txt';
            $match_deadline_hours = 1; // Valeur par défaut : 1 heure
            if (file_exists($match_deadline_hours_file)) {
                $match_deadline_hours = (int)trim(file_get_contents($match_deadline_hours_file));
                if ($match_deadline_hours <= 0) $match_deadline_hours = 1; // Minimum 1 heure
            }
            
            // Charger les barèmes de points
            $points_score_exact = 10;
            $points_bon_vainqueur = 5;
            if (file_exists(__DIR__.'/../includes/points_score_exact.txt')) {
                $points_score_exact = (int)trim(file_get_contents(__DIR__.'/../includes/points_score_exact.txt'));
            }
            if (file_exists(__DIR__.'/../includes/points_bon_vainqueur.txt')) {
                $points_bon_vainqueur = (int)trim(file_get_contents(__DIR__.'/../includes/points_bon_vainqueur.txt'));
            }
            
            // Calculer la deadline : X heures avant le match
            $deadline = date('Y-m-d H:i:s', strtotime($m['match_date']) - ($match_deadline_hours * 3600));
            $disabled = ($m['status'] === 'completed' || date('Y-m-d H:i:s') >= $deadline) ? 'disabled' : '';
            
            $has_prono = (isset($m['predicted_team_1']) && $m['predicted_team_1'] !== '' && $m['predicted_team_1'] !== null)
                || (isset($m['predicted_team_2']) && $m['predicted_team_2'] !== '' && $m['predicted_team_2'] !== null);
            $btn_label = $has_prono ? __('modify_score') : __('validate_my_score');
            $btn_validated_class = $has_prono ? ' btn-validated' : '';
            $btn_validated_check_style = $has_prono ? 'margin-left: 8px; font-weight: bold; font-size: 1.2em;' : 'display: none; margin-left: 8px; font-weight: bold; font-size: 1.2em;';
            $btn_validated_text = $has_prono ? __('validated') : $btn_label;

            // --- HTML DESKTOP (date/heure centrées en haut du cadre) ---
            $html .= '<div class="row match-row p-3 d-none d-md-flex" style="margin-bottom:32px;">';
            $html .= '<div class="w-100 text-center mb-2" style="font-size:1.1rem;">'.htmlspecialchars($date ?? '').'</div>';
            $html .= '<div class="row mb-2" style="font-size:1.1rem;">';
            $html .= '<div class="col-3"></div>';
            $html .= '<div class="col-3 text-center text-muted">'.(($m['status'] !== 'completed' && !$disabled) ? __('enter_my_prediction') : __('my_prediction')).'</div>';
            $html .= '<div class="col-2 text-center text-muted">'.__('real_score').'</div>';
            $html .= '<div class="col-2 text-center text-muted">'.__('my_points').'</div>';
            if ($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc') {
                $html .= '<div class="col-2 text-center text-muted">'.__('starting_eleven').'</div>';
            }
            $html .= '</div>';

            // Préparer les données des équipes
            $equipes = [
                [
                    'flag' => $team1Flag,
                    'name' => getTeamName($m['team_1'], $current_lang),
                    'prono' => $m['predicted_team_1'],
                    'real' => $m['result_team_1']
                ],
                [
                    'flag' => $team2Flag,
                    'name' => getTeamName($m['team_2'], $current_lang),
                    'prono' => $m['predicted_team_2'],
                    'real' => $m['result_team_2']
                ]
            ];

            // Deux lignes pour les équipes
            foreach ($equipes as $i => $equipe) {
                $html .= '<div class="row align-items-center mb-1">';
                // Bloc équipe
                $html .= '<div class="col-3 d-flex align-items-center">';
                $html .= '<img src="'.$equipe['flag'].'" alt="'.htmlspecialchars($equipe['name'] ?? '').'" style="width:32px;height:24px;border-radius:4px;margin-right:8px; margin-left:8px;">';
                $html .= '<span class="fw-bold team-name">'.htmlspecialchars($equipe['name'] ?? '').'</span>';
                // Correction : afficher l'icône 11 titulaires uniquement pour le Maroc
                $match_time = strtotime($m['match_date']);
                $deadline_titulaires = $match_time - 2*3600;
                $now = time();
                if (($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc') && $equipe['name'] === getTeamName('Maroc', $current_lang) && $now < $deadline_titulaires) {
                    $html .= '<span class="select-titulaires" data-match="'.$m['id'].'" style="cursor:pointer;margin-left:8px;" title="'.__('select_starting_eleven').'">';
                    $html .= '<img src="assets/img/11-Titulaire.png" style="height:28px;vertical-align:middle;">';
                    $html .= '</span>';
                }
                $html .= '</div>';
                // Bloc pronostic
                $html .= '<div class="col-3 text-center">';
                if($m['status'] !== 'completed' && !$disabled) {
                    $html .= '<input type="number" min="0" max="20" class="form-control text-center d-inline-block" name="score_'.$m['id'].'_'.($i+1).'" value="'.($equipe['prono'] ?? '').'" style="width:60px;display:inline-block;" oninput="this.value = Math.abs(this.value)">';
                } else {
                    $html .= '<span class="fw-bold">'.($equipe['prono'] ?? '-').'</span>';
                }
                $html .= '</div>';
                // Bloc score réel
                $html .= '<div class="col-2 text-center">';
                if($m['status'] === 'completed') {
                    $html .= '<span class="fw-bold">'.($equipe['real'] ?? '-').'</span>';
                } else {
                    $html .= '<span class="fw-bold">-</span>';
                }
                $html .= '</div>';
                // Bloc points (seulement sur la première ligne)
                if($i === 0) {
                    $html .= '<div class="col-2 text-center" rowspan="2">';
                    if($m['status'] === 'completed') {
                        $points = 0;
                        if (isset($m['result_team_1'], $m['result_team_2'], $m['predicted_team_1'], $m['predicted_team_2'])) {
                            $score1 = intval($m['result_team_1']);
                            $score2 = intval($m['result_team_2']);
                            $p1 = intval($m['predicted_team_1']);
                            $p2 = intval($m['predicted_team_2']);
                            $winner = ($score1 === $score2) ? 'draw' : (($score1 > $score2) ? 'team1' : 'team2');
                            $pred_winner = ($p1 === $p2) ? 'draw' : (($p1 > $p2) ? 'team1' : 'team2');
                            $score_exact = ($p1 === $score1 && $p2 === $score2);
                            $bon_vainqueur = ($pred_winner === $winner && $winner !== 'draw');
                            $is_draw = ($winner === 'draw');
                            if ($is_draw && $pred_winner === 'draw') {
                                $points += $points_bon_vainqueur;
                                if ($score_exact) {
                                    $points += $points_score_exact;
                                }
                            } else {
                                if ($score_exact) {
                                    $points += $points_score_exact;
                                }
                                if ($bon_vainqueur) {
                                    $points += $points_bon_vainqueur;
                                }
                            }
                        }
                        $html .= '<span class="score-value fw-bold text-success" style="font-size:1.2rem;">'.$points.'</span>';
                    } else {
                        $html .= '<span class="score-value text-muted">-</span>';
                    }
                    $html .= '</div>';
                }
                // Après la colonne 'Mes points', dans la boucle d'affichage des équipes, ajoute :
                if($i === 0 && ($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc')) {
                    $html .= '<div class="col-2 text-center" rowspan="2">';
                    if($m['status'] === 'completed') {
                        // Récupérer les points 11 titulaires pour ce match et cet utilisateur
                        $stmt_tit = $mysqli->prepare('SELECT titulaire_11 FROM match_points WHERE user_id = ? AND match_id = ?');
                        $stmt_tit->bind_param('ii', $user_id, $m['id']);
                        $stmt_tit->execute();
                        $res_tit = $stmt_tit->get_result();
                        $row_tit = $res_tit->fetch_assoc();
                        $titulaire_11 = $row_tit ? intval($row_tit['titulaire_11']) : 0;
                        $stmt_tit->close();
                        $html .= '<span class="score-value fw-bold text-primary" style="font-size:1.2rem;">'.$titulaire_11.'</span>';
                    } else {
                        $html .= '<span class="score-value text-muted">-</span>';
                    }
                    $html .= '</div>';
                }
                $html .= '</div>';
            }
            // Bouton Valider sous la grille, dans le même match-row
            if($m['status'] !== 'completed' && !$disabled) {
                $html .= '<div class="col-12 text-center">';
                $html .= '<button type="button" class="btn btn-success mt-2 save-prediction'.$btn_validated_class.'" data-match="'.$m['id'].'" style="min-width: 170px; padding: 2px;">';
                $html .= '<span class="btn-text-prediction">'.$btn_validated_text.'</span>';
                $html .= '<span class="btn-check-prediction" style="'.$btn_validated_check_style.'">✓</span>';
                $html .= '</button>';
                $html .= '</div>';
            }

            $html .= '</div>'; // match-row desktop

            // --- HTML MOBILE (icône 11 titulaires à côté du drapeau Maroc, team_1 ou team_2) ---
            $html .= '<div class="match-card-mobile d-block d-md-none mb-3 p-2" style="background:#23232b;border-radius:12px;box-shadow:0 2px 8px #0002; color: white;">'
                .'<h4 style="color: white; text-align: center; margin-bottom: 17px;">'.htmlspecialchars($date ?? '').'</h4>';
            $html .= '<div class="d-flex justify-content-between align-items-center mb-2" style="margin-bottom: 17px !important;">';
            // Calculer la deadline pour les titulaires (2 heures avant le match)
            $match_time_mobile = strtotime($m['match_date']);
            $deadline_titulaires_mobile = $match_time_mobile - 2*3600;
            $now_mobile = time();
            // Bloc équipe 1
            $html .= '<span class="d-flex align-items-center">'
                .'<img src="'.$team1Flag.'" alt="'.htmlspecialchars(getTeamName($m['team_1'], $current_lang)).'" style="width:28px;height:20px;border-radius:4px;margin-right:6px;">'
                .htmlspecialchars(getTeamName($m['team_1'], $current_lang));
            if ($m['team_1'] === 'Maroc' && $now_mobile < $deadline_titulaires_mobile) {
                $html .= '<span class="select-titulaires" data-match="'.$m['id'].'" style="cursor:pointer;margin-left:8px;" title="'.__('select_starting_eleven').'">';
                $html .= '<img src="assets/img/11-Titulaire.png" style="height:28px;vertical-align:middle;">';
                $html .= '</span>';
            }
            $html .= '</span>';
            // Bloc équipe 2
            $html .= '<span class="d-flex align-items-center">'
                .'<img src="'.$team2Flag.'" alt="'.htmlspecialchars(getTeamName($m['team_2'], $current_lang)).'" style="width:28px;height:20px;border-radius:4px;margin-right:6px;">'
                .'<span class="team-name">'.htmlspecialchars(getTeamName($m['team_2'], $current_lang)).'</span>';
            if ($m['team_2'] === 'Maroc' && $now_mobile < $deadline_titulaires_mobile) {
                $html .= '<span class="select-titulaires" data-match="'.$m['id'].'" style="cursor:pointer;margin-left:8px;" title="'.__('select_starting_eleven').'">';
                $html .= '<img src="assets/img/11-Titulaire.png" style="height:28px;vertical-align:middle;">';
                $html .= '</span>';
            }
            $html .= '</span>';
            $html .= '</div>';
            $html .= '<div class="mb-1"><b>'.__('my_prediction').' :</b> '
                .(($m['status'] !== 'completed' && !$disabled)
                    ? '<input type="number" min="0" max="20" class="form-control d-inline-block text-center" name="score_'.$m['id'].'_1" value="'.($m['predicted_team_1'] ?? '').'" style="width:38px;display:inline-block;">'
                     .' - '
                     .'<input type="number" min="0" max="20" class="form-control d-inline-block text-center" name="score_'.$m['id'].'_2" value="'.($m['predicted_team_2'] ?? '').'" style="width:38px;display:inline-block;">'
                    : '<span class="fw-bold">'.($m['predicted_team_1'] ?? '-').'</span> - <span class="fw-bold">'.($m['predicted_team_2'] ?? '-').'</span>'
                )
            .'</div>';
            $html .= '<div class="mb-1"><b>'.__('real_score').' :</b> '
                .(($m['status'] === 'completed')
                    ? '<span class="fw-bold">'.($m['result_team_1'] ?? '-').'</span> - <span class="fw-bold">'.($m['result_team_2'] ?? '-').'</span>'
                    : '<span class="fw-bold">-</span> - <span class="fw-bold">-</span>'
                )
            .'</div>';
            // Points pronostics
            $points = '-';
            if($m['status'] === 'completed' && isset($m['result_team_1'], $m['result_team_2'], $m['predicted_team_1'], $m['predicted_team_2'])) {
                $score1 = intval($m['result_team_1']);
                $score2 = intval($m['result_team_2']);
                $p1 = intval($m['predicted_team_1']);
                $p2 = intval($m['predicted_team_2']);
                $winner = ($score1 === $score2) ? 'draw' : (($score1 > $score2) ? 'team1' : 'team2');
                $pred_winner = ($p1 === $p2) ? 'draw' : (($p1 > $p2) ? 'team1' : 'team2');
                $score_exact = ($p1 === $score1 && $p2 === $score2);
                $bon_vainqueur = ($pred_winner === $winner && $winner !== 'draw');
                $is_draw = ($winner === 'draw');
                $points = 0;
                if ($is_draw && $pred_winner === 'draw') {
                    $points += $points_bon_vainqueur;
                    if ($score_exact) {
                        $points += $points_score_exact;
                    }
                } else {
                    if ($score_exact) {
                        $points += $points_score_exact;
                    }
                    if ($bon_vainqueur) {
                        $points += $points_bon_vainqueur;
                    }
                }
            }
            $html .= '<div class="mb-1"><b>'.__('my_points').' :</b> <span class="fw-bold text-success">'.$points.'</span></div>';
            // Points titulaires (mobile)
            if($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc') {
                $titulaire_11 = '-';
                if($m['status'] === 'completed') {
                    $stmt_tit = $mysqli->prepare('SELECT titulaire_11 FROM match_points WHERE user_id = ? AND match_id = ?');
                    $stmt_tit->bind_param('ii', $user_id, $m['id']);
                    $stmt_tit->execute();
                    $res_tit = $stmt_tit->get_result();
                    $row_tit = $res_tit->fetch_assoc();
                    $titulaire_11 = $row_tit ? intval($row_tit['titulaire_11']) : 0;
                    $stmt_tit->close();
                }
                $html .= '<div class="mb-1"><b>'.__('starting_eleven').' :</b> <span class="fw-bold text-primary">'.$titulaire_11.'</span></div>';
            }
            // Bouton valider
            if($m['status'] !== 'completed' && !$disabled) {
                $html .= '<div class="text-center mt-2"><button type="button" class="btn btn-success save-prediction'.$btn_validated_class.'" data-match="'.$m['id'].'" style="min-width: 120px;">';
                $html .= '<span class="btn-text-prediction">'.$btn_validated_text.'</span>';
                $html .= '<span class="btn-check-prediction" style="'.$btn_validated_check_style.'">✓</span>';
                $html .= '</button></div>';
            }
            $html .= '</div>'; // match-card-mobile
        }
        $html .= '</div>';
    }
} else {
    // Afficher un message si aucune journée de groupes n'a de matchs
    $html .= '<div class="alert alert-info text-center">'.__('coming_soon').'</div>';
}
$html .= '</div>'; // Fin du bloc phase-block pour groupes
// Autres phases (1 bloc par phase)
// Mapping des phases traduites vers les phases françaises pour data-phase
$phase_mapping = [
    __('group_phase') => 'Phase de groupes',
    __('round_of_16') => '8e de finale',
    __('quarter_final') => 'Quart de finale',
    __('semi_final') => 'Demi-finale',
    __('third_place_match') => 'Match pour la 3e place',
    __('final') => 'Finale'
];

// Afficher d'abord les phases officielles
foreach($phases_officielles as $phase) {
    $phaseMatches = $other_phases[$phase] ?? [];
    // Utiliser la phase traduite pour data-phase (pour correspondre au JavaScript)
    $html .= '<div class="phase-block" data-phase="'.htmlspecialchars($phase ?? '').'">';
    $html .= '<h4 class="mb-2" style="color: #ffb300; visibility: hidden;">'.htmlspecialchars($phase ?? '').'</h4>';
    if (empty($phaseMatches)) {
        $html .= '<div class="alert alert-info text-center">'.__('coming_soon').'</div>';
    } else {
        foreach($phaseMatches as $m) {
            $team1Flag = getFlagUrl($m['team_1']);
            $team2Flag = getFlagUrl($m['team_2']);
            $date = date('d/m/Y H:i', strtotime($m['match_date'])) ?: '';
            
            // Charger la configuration du délai de clôture des pronostics de matchs
            $match_deadline_hours_file = __DIR__.'/../includes/match_deadline_hours.txt';
            $match_deadline_hours = 1; // Valeur par défaut : 1 heure
            if (file_exists($match_deadline_hours_file)) {
                $match_deadline_hours = (int)trim(file_get_contents($match_deadline_hours_file));
                if ($match_deadline_hours <= 0) $match_deadline_hours = 1; // Minimum 1 heure
            }
            
            // Charger les barèmes de points
            $points_score_exact = 10;
            $points_bon_vainqueur = 5;
            if (file_exists(__DIR__.'/../includes/points_score_exact.txt')) {
                $points_score_exact = (int)trim(file_get_contents(__DIR__.'/../includes/points_score_exact.txt'));
            }
            if (file_exists(__DIR__.'/../includes/points_bon_vainqueur.txt')) {
                $points_bon_vainqueur = (int)trim(file_get_contents(__DIR__.'/../includes/points_bon_vainqueur.txt'));
            }
            
            // Calculer la deadline : X heures avant le match
            $deadline = date('Y-m-d H:i:s', strtotime($m['match_date']) - ($match_deadline_hours * 3600));
            $disabled = ($m['status'] === 'completed' || date('Y-m-d H:i:s') >= $deadline) ? 'disabled' : '';
            
            $has_prono = (isset($m['predicted_team_1']) && $m['predicted_team_1'] !== '' && $m['predicted_team_1'] !== null)
                || (isset($m['predicted_team_2']) && $m['predicted_team_2'] !== '' && $m['predicted_team_2'] !== null);
            $btn_label = $has_prono ? __('modify_score') : __('validate_my_score');
            $btn_validated_class = $has_prono ? ' btn-validated' : '';
            $btn_validated_check_style = $has_prono ? 'margin-left: 8px; font-weight: bold; font-size: 1.2em;' : 'display: none; margin-left: 8px; font-weight: bold; font-size: 1.2em;';
            $btn_validated_text = $has_prono ? __('validated') : $btn_label;

            // --- HTML DESKTOP (inchangé, visible md+) ---
            $html .= '<div class="row match-row p-3 d-none d-md-flex" style="margin-bottom:32px;">';
            $html .= '<div class="w-100 text-center mb-2" style="font-size:1.1rem;">'.htmlspecialchars($date ?? '').'</div>';
            $html .= '<div class="row mb-2" style="font-size:1.1rem;">';
            $html .= '<div class="col-3"></div>';
            $html .= '<div class="col-3 text-center text-muted">'.(($m['status'] !== 'completed' && !$disabled) ? __('enter_my_prediction') : __('my_prediction')).'</div>';
            $html .= '<div class="col-2 text-center text-muted">'.__('real_score').'</div>';
            $html .= '<div class="col-2 text-center text-muted">'.__('my_points').'</div>';
            if ($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc') {
                $html .= '<div class="col-2 text-center text-muted">'.__('starting_eleven').'</div>';
            }
            $html .= '</div>';

            // Préparer les données des équipes
            $equipes = [
                [
                    'flag' => $team1Flag,
                    'name' => getTeamName($m['team_1'], $current_lang),
                    'prono' => $m['predicted_team_1'],
                    'real' => $m['result_team_1']
                ],
                [
                    'flag' => $team2Flag,
                    'name' => getTeamName($m['team_2'], $current_lang),
                    'prono' => $m['predicted_team_2'],
                    'real' => $m['result_team_2']
                ]
            ];

            // Deux lignes pour les équipes
            foreach ($equipes as $i => $equipe) {
                $html .= '<div class="row align-items-center mb-1">';
                // Bloc équipe
                $html .= '<div class="col-3 d-flex align-items-center">';
                $html .= '<img src="'.$equipe['flag'].'" alt="'.htmlspecialchars($equipe['name'] ?? '').'" style="width:32px;height:24px;border-radius:4px;margin-right:8px;">';
                $html .= '<span class="fw-bold team-name">'.htmlspecialchars($equipe['name'] ?? '').'</span>';
                // Correction : afficher l'icône 11 titulaires uniquement pour le Maroc
                $match_time = strtotime($m['match_date']);
                $deadline_titulaires = $match_time - 2*3600;
                $now = time();
                if (($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc') && $equipe['name'] === getTeamName('Maroc', $current_lang) && $now < $deadline_titulaires) {
                    $html .= '<span class="select-titulaires" data-match="'.$m['id'].'" style="cursor:pointer;margin-left:8px;" title="'.__('select_starting_eleven').'">';
                    $html .= '<img src="assets/img/11-Titulaire.png" style="height:28px;vertical-align:middle;">';
                    $html .= '</span>';
                }
                $html .= '</div>';
                // Bloc pronostic
                $html .= '<div class="col-3 text-center">';
                if($m['status'] !== 'completed' && !$disabled) {
                    $html .= '<input type="number" min="0" max="20" class="form-control text-center d-inline-block" name="score_'.$m['id'].'_'.($i+1).'" value="'.($equipe['prono'] ?? '').'" style="width:60px;display:inline-block;" oninput="this.value = Math.abs(this.value)">';
                } else {
                    $html .= '<span class="fw-bold">'.($equipe['prono'] ?? '-').'</span>';
                }
                $html .= '</div>';
                // Bloc score réel
                $html .= '<div class="col-2 text-center">';
                if($m['status'] === 'completed') {
                    $html .= '<span class="fw-bold">'.($equipe['real'] ?? '-').'</span>';
                } else {
                    $html .= '<span class="fw-bold">-</span>';
                }
                $html .= '</div>';
                // Bloc points (seulement sur la première ligne)
                if($i === 0) {
                    $html .= '<div class="col-2 text-center" rowspan="2">';
                    if($m['status'] === 'completed') {
                        $points = 0;
                        if (isset($m['result_team_1'], $m['result_team_2'], $m['predicted_team_1'], $m['predicted_team_2'])) {
                            $score1 = intval($m['result_team_1']);
                            $score2 = intval($m['result_team_2']);
                            $p1 = intval($m['predicted_team_1']);
                            $p2 = intval($m['predicted_team_2']);
                            $winner = ($score1 === $score2) ? 'draw' : (($score1 > $score2) ? 'team1' : 'team2');
                            $pred_winner = ($p1 === $p2) ? 'draw' : (($p1 > $p2) ? 'team1' : 'team2');
                            $score_exact = ($p1 === $score1 && $p2 === $score2);
                            $bon_vainqueur = ($pred_winner === $winner && $winner !== 'draw');
                            $is_draw = ($winner === 'draw');
                            if ($is_draw && $pred_winner === 'draw') {
                                $points += $points_bon_vainqueur;
                                if ($score_exact) {
                                    $points += $points_score_exact;
                                }
                            } else {
                                if ($score_exact) {
                                    $points += $points_score_exact;
                                }
                                if ($bon_vainqueur) {
                                    $points += $points_bon_vainqueur;
                                }
                            }
                        }
                        $html .= '<span class="score-value fw-bold text-success" style="font-size:1.2rem;">'.$points.'</span>';
                    } else {
                        $html .= '<span class="score-value text-muted">-</span>';
                    }
                    $html .= '</div>';
                }
                // Après la colonne 'Mes points', dans la boucle d'affichage des équipes, ajoute :
                if($i === 0 && ($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc')) {
                    $html .= '<div class="col-2 text-center" rowspan="2">';
                    if($m['status'] === 'completed') {
                        // Récupérer les points 11 titulaires pour ce match et cet utilisateur
                        $stmt_tit = $mysqli->prepare('SELECT titulaire_11 FROM match_points WHERE user_id = ? AND match_id = ?');
                        $stmt_tit->bind_param('ii', $user_id, $m['id']);
                        $stmt_tit->execute();
                        $res_tit = $stmt_tit->get_result();
                        $row_tit = $res_tit->fetch_assoc();
                        $titulaire_11 = $row_tit ? intval($row_tit['titulaire_11']) : 0;
                        $stmt_tit->close();
                        $html .= '<span class="score-value fw-bold text-primary" style="font-size:1.2rem;">'.$titulaire_11.'</span>';
                    } else {
                        $html .= '<span class="score-value text-muted">-</span>';
                    }
                    $html .= '</div>';
                }
                $html .= '</div>';
            }
            // Bouton Valider sous la grille, dans le même match-row
            if($m['status'] !== 'completed' && !$disabled) {
                $html .= '<div class="col-12 text-center">';
                $html .= '<button type="button" class="btn btn-success mt-2 save-prediction'.$btn_validated_class.'" data-match="'.$m['id'].'" style="min-width: 170px; padding: 2px;">';
                $html .= '<span class="btn-text-prediction">'.$btn_validated_text.'</span>';
                $html .= '<span class="btn-check-prediction" style="'.$btn_validated_check_style.'">✓</span>';
                $html .= '</button>';
                $html .= '</div>';
            }

            $html .= '</div>'; // match-row desktop

            // --- HTML MOBILE (ajouté) ---
            $html .= '<div class="match-card-mobile d-block d-md-none mb-3 p-2" style="background:#23232b;border-radius:12px;box-shadow:0 2px 8px #0002; color: white;">';
            $html .= '<h4 style="color: white; text-align: center; margin-bottom: 17px;">'.htmlspecialchars($date).'</h4>';
            $html .= '<div class="d-flex justify-content-between align-items-center mb-2" style="margin-bottom: 17px !important;">';
            $html .= '<span class="d-flex align-items-center">'
                .'<img src="'.$team1Flag.'" alt="'.htmlspecialchars(getTeamName($m['team_1'], $current_lang)).'" style="width:28px;height:20px;border-radius:4px;margin-right:6px;">'
                .'<span class="team-name">'.htmlspecialchars(getTeamName($m['team_1'], $current_lang)).'</span>';
            $html .= '</span>';
            $html .= '<span class="fw-bold text-warning">vs</span>';
            $html .= '<span class="d-flex align-items-center">'
                .'<img src="'.$team2Flag.'" alt="'.htmlspecialchars(getTeamName($m['team_2'], $current_lang)).'" style="width:28px;height:20px;border-radius:4px;margin-right:6px;">'
                .'<span class="team-name">'.htmlspecialchars(getTeamName($m['team_2'], $current_lang)).'</span></span>';
            $html .= '</div>';
            $html .= '<div class="mb-1"><b>'.__('my_prediction').' :</b> '
                .(($m['status'] !== 'completed' && !$disabled)
                    ? '<input type="number" min="0" max="20" class="form-control d-inline-block text-center" name="score_'.$m['id'].'_1" value="'.($m['predicted_team_1'] ?? '').'" style="width:38px;display:inline-block;">'
                     .' - '
                     .'<input type="number" min="0" max="20" class="form-control d-inline-block text-center" name="score_'.$m['id'].'_2" value="'.($m['predicted_team_2'] ?? '').'" style="width:38px;display:inline-block;">'
                    : '<span class="fw-bold">'.($m['predicted_team_1'] ?? '-').'</span> - <span class="fw-bold">'.($m['predicted_team_2'] ?? '-').'</span>'
                )
            .'</div>';
            $html .= '<div class="mb-1"><b>'.__('real_score').' :</b> '
                .(($m['status'] === 'completed')
                    ? '<span class="fw-bold">'.($m['result_team_1'] ?? '-').'</span> - <span class="fw-bold">'.($m['result_team_2'] ?? '-').'</span>'
                    : '<span class="fw-bold">-</span> - <span class="fw-bold">-</span>'
                )
            .'</div>';
            // Points pronostics
            $points = '-';
            if($m['status'] === 'completed' && isset($m['result_team_1'], $m['result_team_2'], $m['predicted_team_1'], $m['predicted_team_2'])) {
                $score1 = intval($m['result_team_1']);
                $score2 = intval($m['result_team_2']);
                $p1 = intval($m['predicted_team_1']);
                $p2 = intval($m['predicted_team_2']);
                $winner = ($score1 === $score2) ? 'draw' : (($score1 > $score2) ? 'team1' : 'team2');
                $pred_winner = ($p1 === $p2) ? 'draw' : (($p1 > $p2) ? 'team1' : 'team2');
                $score_exact = ($p1 === $score1 && $p2 === $score2);
                $bon_vainqueur = ($pred_winner === $winner && $winner !== 'draw');
                $is_draw = ($winner === 'draw');
                $points = 0;
                if ($is_draw && $pred_winner === 'draw') {
                    $points += $points_bon_vainqueur;
                    if ($score_exact) {
                        $points += $points_score_exact;
                    }
                } else {
                    if ($score_exact) {
                        $points += $points_score_exact;
                    }
                    if ($bon_vainqueur) {
                        $points += $points_bon_vainqueur;
                    }
                }
            }
            $html .= '<div class="mb-1"><b>'.__('my_points').' :</b> <span class="fw-bold text-success">'.$points.'</span></div>';
            // Points titulaires (mobile)
            if($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc') {
                $titulaire_11 = '-';
                if($m['status'] === 'completed') {
                    $stmt_tit = $mysqli->prepare('SELECT titulaire_11 FROM match_points WHERE user_id = ? AND match_id = ?');
                    $stmt_tit->bind_param('ii', $user_id, $m['id']);
                    $stmt_tit->execute();
                    $res_tit = $stmt_tit->get_result();
                    $row_tit = $res_tit->fetch_assoc();
                    $titulaire_11 = $row_tit ? intval($row_tit['titulaire_11']) : 0;
                    $stmt_tit->close();
                }
                $html .= '<div class="mb-1"><b>'.__('starting_eleven').' :</b> <span class="fw-bold text-primary">'.$titulaire_11.'</span></div>';
            }
            // Bouton valider
            if($m['status'] !== 'completed' && !$disabled) {
                $html .= '<div class="text-center mt-2"><button type="button" class="btn btn-success save-prediction'.$btn_validated_class.'" data-match="'.$m['id'].'" style="min-width: 120px;">';
                $html .= '<span class="btn-text-prediction">'.$btn_validated_text.'</span>';
                $html .= '<span class="btn-check-prediction" style="'.$btn_validated_check_style.'">✓</span>';
                $html .= '</button></div>';
            }
            $html .= '</div>'; // match-card-mobile
        }
    }
    $html .= '</div>';
}

// Afficher toutes les autres phases qui ne sont pas dans la liste officielle
foreach($other_phases as $phase => $phaseMatches) {
    // Vérifier si cette phase n'a pas déjà été affichée dans les phases officielles
    if (!in_array($phase, $phases_officielles)) {
        $html .= '<div class="phase-block" data-phase="'.htmlspecialchars($phase ?? '').'">';
        $html .= '<h4 class="mb-2" style="color: #ffb300; visibility: hidden;">'.htmlspecialchars($phase ?? '').'</h4>';
        if (empty($phaseMatches)) {
            $html .= '<div class="alert alert-info text-center">'.__('coming_soon').'</div>';
        } else {
            // Utiliser le même code d'affichage que pour les phases officielles
            foreach($phaseMatches as $m) {
                $team1Flag = getFlagUrl($m['team_1']);
                $team2Flag = getFlagUrl($m['team_2']);
                $date = date('d/m/Y H:i', strtotime($m['match_date'])) ?: '';
                
                // Charger la configuration du délai de clôture des pronostics de matchs
                $match_deadline_hours_file = __DIR__.'/../includes/match_deadline_hours.txt';
                $match_deadline_hours = 1; // Valeur par défaut : 1 heure
                if (file_exists($match_deadline_hours_file)) {
                    $match_deadline_hours = (int)trim(file_get_contents($match_deadline_hours_file));
                    if ($match_deadline_hours <= 0) $match_deadline_hours = 1; // Minimum 1 heure
                }
                
                // Charger les barèmes de points
                $points_score_exact = 10;
                $points_bon_vainqueur = 5;
                if (file_exists(__DIR__.'/../includes/points_score_exact.txt')) {
                    $points_score_exact = (int)trim(file_get_contents(__DIR__.'/../includes/points_score_exact.txt'));
                }
                if (file_exists(__DIR__.'/../includes/points_bon_vainqueur.txt')) {
                    $points_bon_vainqueur = (int)trim(file_get_contents(__DIR__.'/../includes/points_bon_vainqueur.txt'));
                }
                
                // Calculer la deadline : X heures avant le match
                $deadline = date('Y-m-d H:i:s', strtotime($m['match_date']) - ($match_deadline_hours * 3600));
                $disabled = ($m['status'] === 'completed' || date('Y-m-d H:i:s') >= $deadline) ? 'disabled' : '';
                
                $has_prono = (isset($m['predicted_team_1']) && $m['predicted_team_1'] !== '' && $m['predicted_team_1'] !== null)
                    || (isset($m['predicted_team_2']) && $m['predicted_team_2'] !== '' && $m['predicted_team_2'] !== null);
                $btn_label = $has_prono ? __('modify_score') : __('validate_my_score');
                $btn_validated_class = $has_prono ? ' btn-validated' : '';
                $btn_validated_check_style = $has_prono ? 'margin-left: 8px; font-weight: bold; font-size: 1.2em;' : 'display: none; margin-left: 8px; font-weight: bold; font-size: 1.2em;';
                $btn_validated_text = $has_prono ? __('validated') : $btn_label;

                // HTML DESKTOP
                $html .= '<div class="row match-row p-3 d-none d-md-flex" style="margin-bottom:32px;">';
                $html .= '<div class="w-100 text-center mb-2" style="font-size:1.1rem;">'.htmlspecialchars($date ?? '').'</div>';
                $html .= '<div class="row mb-2" style="font-size:1.1rem;">';
                $html .= '<div class="col-3"></div>';
                $html .= '<div class="col-3 text-center text-muted">'.(($m['status'] !== 'completed' && !$disabled) ? __('enter_my_prediction') : __('my_prediction')).'</div>';
                $html .= '<div class="col-2 text-center text-muted">'.__('real_score').'</div>';
                $html .= '<div class="col-2 text-center text-muted">'.__('my_points').'</div>';
                if ($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc') {
                    $html .= '<div class="col-2 text-center text-muted">'.__('starting_eleven').'</div>';
                }
                $html .= '</div>';

                // Préparer les données des équipes
                $equipes = [
                    [
                        'flag' => $team1Flag,
                        'name' => getTeamName($m['team_1'], $current_lang),
                        'prono' => $m['predicted_team_1'],
                        'real' => $m['result_team_1']
                    ],
                    [
                        'flag' => $team2Flag,
                        'name' => getTeamName($m['team_2'], $current_lang),
                        'prono' => $m['predicted_team_2'],
                        'real' => $m['result_team_2']
                    ]
                ];

                // Deux lignes pour les équipes
                foreach ($equipes as $i => $equipe) {
                    $html .= '<div class="row align-items-center mb-1">';
                    // Bloc équipe
                    $html .= '<div class="col-3 d-flex align-items-center">';
                    $html .= '<img src="'.$equipe['flag'].'" alt="'.htmlspecialchars($equipe['name'] ?? '').'" style="width:32px;height:24px;border-radius:4px;margin-right:8px; margin-left:8px;">';
                    $html .= '<span class="fw-bold team-name">'.htmlspecialchars($equipe['name'] ?? '').'</span>';
                    $match_time = strtotime($m['match_date']);
                    $deadline_titulaires = $match_time - 2*3600;
                    $now = time();
                    if (($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc') && $equipe['name'] === getTeamName('Maroc', $current_lang) && $now < $deadline_titulaires) {
                        $html .= '<span class="select-titulaires" data-match="'.$m['id'].'" style="cursor:pointer;margin-left:8px;" title="'.__('select_starting_eleven').'">';
                        $html .= '<img src="assets/img/11-Titulaire.png" style="height:28px;vertical-align:middle;">';
                        $html .= '</span>';
                    }
                    $html .= '</div>';
                    // Bloc pronostic
                    $html .= '<div class="col-3 text-center">';
                    if($m['status'] !== 'completed' && !$disabled) {
                        $html .= '<input type="number" min="0" max="20" class="form-control text-center d-inline-block" name="score_'.$m['id'].'_'.($i+1).'" value="'.($equipe['prono'] ?? '').'" style="width:60px;display:inline-block;" oninput="this.value = Math.abs(this.value)">';
                    } else {
                        $html .= '<span class="fw-bold">'.($equipe['prono'] ?? '-').'</span>';
                    }
                    $html .= '</div>';
                    // Bloc score réel
                    $html .= '<div class="col-2 text-center">';
                    if($m['status'] === 'completed') {
                        $html .= '<span class="fw-bold">'.($equipe['real'] ?? '-').'</span>';
                    } else {
                        $html .= '<span class="fw-bold">-</span>';
                    }
                    $html .= '</div>';
                    // Bloc points (seulement sur la première ligne)
                    if($i === 0) {
                        $html .= '<div class="col-2 text-center" rowspan="2">';
                        if($m['status'] === 'completed') {
                            $points = 0;
                            if (isset($m['result_team_1'], $m['result_team_2'], $m['predicted_team_1'], $m['predicted_team_2'])) {
                                $score1 = intval($m['result_team_1']);
                                $score2 = intval($m['result_team_2']);
                                $p1 = intval($m['predicted_team_1']);
                                $p2 = intval($m['predicted_team_2']);
                                $winner = ($score1 === $score2) ? 'draw' : (($score1 > $score2) ? 'team1' : 'team2');
                                $pred_winner = ($p1 === $p2) ? 'draw' : (($p1 > $p2) ? 'team1' : 'team2');
                                $score_exact = ($p1 === $score1 && $p2 === $score2);
                                $bon_vainqueur = ($pred_winner === $winner && $winner !== 'draw');
                                $is_draw = ($winner === 'draw');
                                if ($is_draw && $pred_winner === 'draw') {
                                    $points += $points_bon_vainqueur;
                                    if ($score_exact) {
                                        $points += $points_score_exact;
                                    }
                                } else {
                                    if ($score_exact) {
                                        $points += $points_score_exact;
                                    }
                                    if ($bon_vainqueur) {
                                        $points += $points_bon_vainqueur;
                                    }
                                }
                            }
                            $html .= '<span class="badge bg-success">'.$points.'</span>';
                        } else {
                            $html .= '<span class="badge bg-secondary">-</span>';
                        }
                        $html .= '</div>';
                        // Bloc titulaires (seulement sur la première ligne)
                        if ($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc') {
                            $html .= '<div class="col-2 text-center" rowspan="2">';
                            if($m['status'] === 'completed') {
                                $stmt_tit = $mysqli->prepare('SELECT titulaire_11 FROM match_points WHERE user_id = ? AND match_id = ?');
                                $stmt_tit->bind_param('ii', $user_id, $m['id']);
                                $stmt_tit->execute();
                                $res_tit = $stmt_tit->get_result();
                                $row_tit = $res_tit->fetch_assoc();
                                $titulaire_11 = $row_tit ? intval($row_tit['titulaire_11']) : 0;
                                $stmt_tit->close();
                                $html .= '<span class="badge bg-primary">'.$titulaire_11.'</span>';
                            } else {
                                $html .= '<span class="badge bg-secondary">-</span>';
                            }
                            $html .= '</div>';
                        }
                    }
                    $html .= '</div>';
                }
                // Bouton valider
                if($m['status'] !== 'completed' && !$disabled) {
                    $html .= '<div class="row mt-2"><div class="col-12 text-center">';
                    $html .= '<button type="button" class="btn btn-success save-prediction'.$btn_validated_class.'" data-match="'.$m['id'].'">';
                    $html .= '<span class="btn-text-prediction">'.$btn_validated_text.'</span>';
                    $html .= '<span class="btn-check-prediction" style="'.$btn_validated_check_style.'">✓</span>';
                    $html .= '</button>';
                    $html .= '</div></div>';
                }
                $html .= '</div>'; // match-row desktop

                // HTML MOBILE
                $html .= '<div class="match-card-mobile d-block d-md-none mb-3 p-2" style="background:#23232b;border-radius:12px;box-shadow:0 2px 8px #0002; color: white;">'
                    .'<h4 style="color: white; text-align: center; margin-bottom: 17px;">'.htmlspecialchars($date).'</h4>';
                $html .= '<div class="d-flex justify-content-between align-items-center mb-2" style="margin-bottom: 17px !important;">';
                $html .= '<span class="d-flex align-items-center">'
                    .'<img src="'.$team1Flag.'" alt="'.htmlspecialchars(getTeamName($m['team_1'], $current_lang)).'" style="width:28px;height:20px;border-radius:4px;margin-right:6px;">'
                    .'<span class="team-name">'.htmlspecialchars(getTeamName($m['team_1'], $current_lang)).'</span>';
                $html .= '</span>';
                $html .= '<span class="fw-bold text-warning">vs</span>';
                $html .= '<span class="d-flex align-items-center">'
                    .'<img src="'.$team2Flag.'" alt="'.htmlspecialchars(getTeamName($m['team_2'], $current_lang)).'" style="width:28px;height:20px;border-radius:4px;margin-right:6px;">'
                    .'<span class="team-name">'.htmlspecialchars(getTeamName($m['team_2'], $current_lang)).'</span></span>';
                $html .= '</div>';
                $html .= '<div class="mb-1"><b>'.__('my_prediction').' :</b> '
                    .(($m['status'] !== 'completed' && !$disabled)
                        ? '<input type="number" min="0" max="20" class="form-control d-inline-block text-center" name="score_'.$m['id'].'_1" value="'.($m['predicted_team_1'] ?? '').'" style="width:38px;display:inline-block;">'
                         .' - '
                         .'<input type="number" min="0" max="20" class="form-control d-inline-block text-center" name="score_'.$m['id'].'_2" value="'.($m['predicted_team_2'] ?? '').'" style="width:38px;display:inline-block;">'
                        : '<span class="fw-bold">'.($m['predicted_team_1'] ?? '-').'</span> - <span class="fw-bold">'.($m['predicted_team_2'] ?? '-').'</span>'
                    )
                .'</div>';
                $html .= '<div class="mb-1"><b>'.__('real_score').' :</b> '
                    .(($m['status'] === 'completed')
                        ? '<span class="fw-bold">'.($m['result_team_1'] ?? '-').'</span> - <span class="fw-bold">'.($m['result_team_2'] ?? '-').'</span>'
                        : '<span class="fw-bold">-</span> - <span class="fw-bold">-</span>'
                    )
                .'</div>';
                // Points pronostics
                $points = '-';
                if($m['status'] === 'completed' && isset($m['result_team_1'], $m['result_team_2'], $m['predicted_team_1'], $m['predicted_team_2'])) {
                    $score1 = intval($m['result_team_1']);
                    $score2 = intval($m['result_team_2']);
                    $p1 = intval($m['predicted_team_1']);
                    $p2 = intval($m['predicted_team_2']);
                    $winner = ($score1 === $score2) ? 'draw' : (($score1 > $score2) ? 'team1' : 'team2');
                    $pred_winner = ($p1 === $p2) ? 'draw' : (($p1 > $p2) ? 'team1' : 'team2');
                    $score_exact = ($p1 === $score1 && $p2 === $score2);
                    $bon_vainqueur = ($pred_winner === $winner && $winner !== 'draw');
                    $is_draw = ($winner === 'draw');
                    $points = 0;
                    if ($is_draw && $pred_winner === 'draw') {
                        $points += $points_bon_vainqueur;
                        if ($score_exact) {
                            $points += $points_score_exact;
                        }
                    } else {
                        if ($score_exact) {
                            $points += $points_score_exact;
                        }
                        if ($bon_vainqueur) {
                            $points += $points_bon_vainqueur;
                        }
                    }
                }
                $html .= '<div class="mb-1"><b>'.__('my_points').' :</b> <span class="fw-bold text-success">'.$points.'</span></div>';
                // Points titulaires (mobile)
                if($m['team_1'] === 'Maroc' || $m['team_2'] === 'Maroc') {
                    $titulaire_11 = '-';
                    if($m['status'] === 'completed') {
                        $stmt_tit = $mysqli->prepare('SELECT titulaire_11 FROM match_points WHERE user_id = ? AND match_id = ?');
                        $stmt_tit->bind_param('ii', $user_id, $m['id']);
                        $stmt_tit->execute();
                        $res_tit = $stmt_tit->get_result();
                        $row_tit = $res_tit->fetch_assoc();
                        $titulaire_11 = $row_tit ? intval($row_tit['titulaire_11']) : 0;
                        $stmt_tit->close();
                    }
                    $html .= '<div class="mb-1"><b>'.__('starting_eleven').' :</b> <span class="fw-bold text-primary">'.$titulaire_11.'</span></div>';
                }
                // Bouton valider
                if($m['status'] !== 'completed' && !$disabled) {
                    $html .= '<div class="text-center mt-2"><button type="button" class="btn btn-success save-prediction'.$btn_validated_class.'" data-match="'.$m['id'].'" style="min-width: 120px;">';
                    $html .= '<span class="btn-text-prediction">'.$btn_validated_text.'</span>';
                    $html .= '<span class="btn-check-prediction" style="'.$btn_validated_check_style.'">✓</span>';
                    $html .= '</button></div>';
                }
                $html .= '</div>'; // match-card-mobile
            }
        }
        $html .= '</div>';
    }
}

// Préparer les traductions des phases pour JavaScript
$phases_titres = [];
// Mapping des phases traduites vers les phases françaises
$phase_fr_mapping = [
    __('group_phase') => 'Phase de groupes',
    __('round_of_16') => '8e de finale',
    __('quarter_final') => 'Quart de finale',
    __('semi_final') => 'Demi-finale',
    __('third_place_match') => 'Match pour la 3e place',
    __('third_place_short') => 'Match 3e place',
    __('final') => 'Finale'
];

$phases_officielles = [
    __('group_phase'),
    __('round_of_16'),
    __('quarter_final'),
    __('semi_final'),
    __('third_place_match'),
    __('third_place_short'),
    __('final')
];

foreach($phases_officielles as $phase) {
    // getPhaseTitre attend le nom français, donc on utilise le mapping
    $phase_fr = $phase_fr_mapping[$phase] ?? $phase;
    $phases_titres[$phase] = getPhaseTitre($phase_fr, $current_lang);
}

// S'assurer que la phase de groupes est bien dans phases_titres
$group_phase_translated = __('group_phase');
if (!isset($phases_titres[$group_phase_translated])) {
    $phases_titres[$group_phase_translated] = getPhaseTitre('Phase de groupes', $current_lang);
}

// Préparer les traductions des journées pour JavaScript
$journees_titres = [];

// Créer un mapping pour toutes les variantes possibles des journées
$journees_variantes = [
    // Français - chaînes complètes (comme elles arrivent de la base de données)
    'Phase de groupes · Journée 1 sur 3',
    'Phase de groupes · Journée 2 sur 3', 
    'Phase de groupes · Journée 3 sur 3',
    // Français - chaînes simples (ce que getJourneeTitre retourne)
    'Journée 1 sur 3',
    'Journée 2 sur 3',
    'Journée 3 sur 3',
    // Arabe - chaînes complètes (si les traductions sont appliquées)
    'مرحلة المجموعات · الجولة 1 من 3',
    'مرحلة المجموعات · الجولة 2 من 3',
    'مرحلة المجموعات · الجولة 3 من 3',
    // Arabe - chaînes simples (ce que getJourneeTitre retourne)
    'الجولة الأولى من 3',
    'الجولة الثانية من 3',
    'الجولة الثالثة من 3'
];

foreach($journees_variantes as $journee) {
    $journees_titres[$journee] = getJourneeTitre($journee, $current_lang);
}

echo json_encode([
    'success' => true, 
    'html' => $html,
    'phases_titres' => $phases_titres,
    'journees_titres' => $journees_titres
]); 