<?php
session_start();
header('Content-Type: application/json');

// Vérifier que l'utilisateur est admin ou stat
if (!isset($_SESSION['role']) || ($_SESSION['role'] !== 'admin' && $_SESSION['role'] !== 'stat')) {
    echo json_encode(['success' => false, 'message' => 'Non autorisé']);
    exit;
}

require_once '../includes/db.php';
require_once '../includes/email.php';

$action = $_POST['action'] ?? '';
$user_id = intval($_POST['user_id'] ?? 0);

try {
    switch ($action) {
        case 'add':
            $username = trim($_POST['username'] ?? '');
            $email = trim($_POST['email'] ?? '');
            $site = trim($_POST['site'] ?? 'Jorf');
            $ville = trim($_POST['ville'] ?? '');
            $departement = trim($_POST['departement'] ?? '');
            
            if (!$username || !$email) {
                echo json_encode(['success' => false, 'message' => 'Nom et email sont obligatoires.']);
                exit;
            }
            
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                echo json_encode(['success' => false, 'message' => 'Format d\'email invalide.']);
                exit;
            }
            
            // Vérifier si l'email existe déjà
            $checkStmt = $mysqli->prepare('SELECT id FROM Users WHERE email = ?');
            $checkStmt->bind_param('s', $email);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();
            
            if ($checkResult->num_rows > 0) {
                echo json_encode(['success' => false, 'message' => 'Cet email est déjà utilisé.']);
                $checkStmt->close();
                exit;
            }
            $checkStmt->close();
            
            // Générer un mot de passe aléatoire
            $generatedPassword = bin2hex(random_bytes(8));
            $hash = password_hash($generatedPassword, PASSWORD_DEFAULT);
            
            // Insérer le nouvel utilisateur
            $stmt = $mysqli->prepare('INSERT INTO Users (username, email, password, site, ville, departement, role) VALUES (?, ?, ?, ?, ?, ?, "user")');
            $stmt->bind_param('ssssss', $username, $email, $hash, $site, $ville, $departement);
            
            if ($stmt->execute()) {
                // Envoyer l'email avec le mot de passe
                $emailSent = sendPasswordEmail($email, $username, $generatedPassword);
                
                $message = 'Utilisateur ajouté avec succès.';
                if ($emailSent) {
                    $message .= ' Un email avec le mot de passe a été envoyé à ' . $email . '.';
                } else {
                    $message .= ' Mot de passe temporaire : ' . $generatedPassword . ' (Notez-le précieusement, l\'email n\'a pas pu être envoyé).';
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => $message,
                    'password' => $generatedPassword,
                    'email_sent' => $emailSent
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Erreur lors de l\'ajout.']);
            }
            $stmt->close();
            break;
            
        case 'edit':
            if (!$user_id) {
                echo json_encode(['success' => false, 'message' => 'ID utilisateur manquant.']);
                exit;
            }
            
            $username = trim($_POST['username'] ?? '');
            $email = trim($_POST['email'] ?? '');
            $site = trim($_POST['site'] ?? '');
            $ville = trim($_POST['ville'] ?? '');
            $departement = trim($_POST['departement'] ?? '');
            
            if (!$username || !$email) {
                echo json_encode(['success' => false, 'message' => 'Nom et email sont obligatoires.']);
                exit;
            }
            
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                echo json_encode(['success' => false, 'message' => 'Format d\'email invalide.']);
                exit;
            }
            
            // Vérifier si l'email est utilisé par un autre utilisateur
            $checkStmt = $mysqli->prepare('SELECT id FROM Users WHERE email = ? AND id != ?');
            $checkStmt->bind_param('si', $email, $user_id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();
            
            if ($checkResult->num_rows > 0) {
                echo json_encode(['success' => false, 'message' => 'Cet email est déjà utilisé par un autre utilisateur.']);
                $checkStmt->close();
                exit;
            }
            $checkStmt->close();
            
            // Mettre à jour l'utilisateur
            $stmt = $mysqli->prepare('UPDATE Users SET username = ?, email = ?, site = ?, ville = ?, departement = ? WHERE id = ?');
            $stmt->bind_param('sssssi', $username, $email, $site, $ville, $departement, $user_id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Utilisateur modifié avec succès.']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Erreur lors de la modification.']);
            }
            $stmt->close();
            break;
            
        case 'change_password':
            if (!$user_id) {
                echo json_encode(['success' => false, 'message' => 'ID utilisateur manquant.']);
                exit;
            }
            
            $new_password = trim($_POST['new_password'] ?? '');
            
            if (!$new_password || strlen($new_password) < 6) {
                echo json_encode(['success' => false, 'message' => 'Le mot de passe doit contenir au moins 6 caractères.']);
                exit;
            }
            
            // Vérifier que l'utilisateur existe
            $checkStmt = $mysqli->prepare('SELECT id FROM Users WHERE id = ?');
            $checkStmt->bind_param('i', $user_id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();
            $user = $checkResult->fetch_assoc();
            $checkStmt->close();
            
            if (!$user) {
                echo json_encode(['success' => false, 'message' => 'Utilisateur non trouvé.']);
                exit;
            }
            
            // Hasher le nouveau mot de passe
            $hash = password_hash($new_password, PASSWORD_DEFAULT);
            
            // Mettre à jour le mot de passe
            $stmt = $mysqli->prepare('UPDATE Users SET password = ? WHERE id = ?');
            $stmt->bind_param('si', $hash, $user_id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Mot de passe modifié avec succès.']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Erreur lors de la modification du mot de passe.']);
            }
            $stmt->close();
            break;
            
        case 'delete':
            if (!$user_id) {
                echo json_encode(['success' => false, 'message' => 'ID utilisateur manquant.']);
                exit;
            }
            
            // Vérifier que ce n'est pas un admin
            $checkStmt = $mysqli->prepare('SELECT role FROM Users WHERE id = ?');
            $checkStmt->bind_param('i', $user_id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();
            $user = $checkResult->fetch_assoc();
            $checkStmt->close();
            
            if (!$user) {
                echo json_encode(['success' => false, 'message' => 'Utilisateur non trouvé.']);
                exit;
            }
            
            if ($user['role'] === 'admin' || $user['role'] === 'stat') {
                echo json_encode(['success' => false, 'message' => 'Impossible de supprimer un administrateur.']);
                exit;
            }
            
            // Supprimer l'utilisateur
            $stmt = $mysqli->prepare('DELETE FROM Users WHERE id = ?');
            $stmt->bind_param('i', $user_id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Utilisateur supprimé avec succès.']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Erreur lors de la suppression.']);
            }
            $stmt->close();
            break;
            
        case 'toggle_block':
            if (!$user_id) {
                echo json_encode(['success' => false, 'message' => 'ID utilisateur manquant.']);
                exit;
            }
            
            // Vérifier si le champ is_blocked existe, sinon le créer
            $checkColumn = $mysqli->query("SHOW COLUMNS FROM Users LIKE 'is_blocked'");
            if ($checkColumn->num_rows == 0) {
                $mysqli->query("ALTER TABLE Users ADD COLUMN is_blocked TINYINT(1) DEFAULT 0");
            }
            
            // Récupérer l'état actuel
            $checkStmt = $mysqli->prepare('SELECT is_blocked FROM Users WHERE id = ?');
            $checkStmt->bind_param('i', $user_id);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();
            $user = $checkResult->fetch_assoc();
            $checkStmt->close();
            
            if (!$user) {
                echo json_encode(['success' => false, 'message' => 'Utilisateur non trouvé.']);
                exit;
            }
            
            // Gérer les valeurs NULL comme 0 (non bloqué)
            $current_status = isset($user['is_blocked']) ? intval($user['is_blocked']) : 0;
            $new_status = ($current_status == 1) ? 0 : 1;
            
            // Mettre à jour le statut
            $stmt = $mysqli->prepare('UPDATE Users SET is_blocked = ? WHERE id = ?');
            $stmt->bind_param('ii', $new_status, $user_id);
            
            if ($stmt->execute()) {
                $status_text = $new_status ? 'bloqué' : 'débloqué';
                echo json_encode([
                    'success' => true,
                    'message' => "Utilisateur $status_text avec succès.",
                    'is_blocked' => $new_status
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Erreur lors de la modification.']);
            }
            $stmt->close();
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Action non reconnue.']);
            break;
    }
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Erreur serveur : ' . $e->getMessage()]);
}

