<?php
header('Content-Type: application/json');
require_once '../includes/db.php';

// Récupérer et nettoyer les données
$token = isset($_POST['token']) ? trim($_POST['token']) : '';
$newPassword = isset($_POST['newPassword']) ? trim($_POST['newPassword']) : '';
$confirmPassword = isset($_POST['confirmPassword']) ? trim($_POST['confirmPassword']) : '';

// Validation des champs requis
if (empty($token) || empty($newPassword) || empty($confirmPassword)) {
    echo json_encode(['success' => false, 'message' => 'Tous les champs sont obligatoires.']);
    exit;
}

// Nettoyer le token (ne garder que les caractères hexadécimaux)
$token = preg_replace('/[^a-f0-9]/i', '', $token);

// Vérifier la longueur du token
if (strlen($token) !== 64) {
    echo json_encode(['success' => false, 'message' => 'Token invalide. Format incorrect.']);
    exit;
}

// Validation des mots de passe
if (strlen($newPassword) < 6) {
    echo json_encode(['success' => false, 'message' => 'Le mot de passe doit contenir au moins 6 caractères.']);
    exit;
}

if ($newPassword !== $confirmPassword) {
    echo json_encode(['success' => false, 'message' => 'Les mots de passe ne correspondent pas.']);
    exit;
}

try {
    // Étape 1: Vérifier si le token existe dans la base
    $tokenStmt = $mysqli->prepare('SELECT user_id, expires_at FROM password_reset_tokens WHERE token = ?');
    
    if (!$tokenStmt) {
        throw new Exception('Erreur de préparation de la requête token: ' . $mysqli->error);
    }
    
    $tokenStmt->bind_param('s', $token);
    $tokenStmt->execute();
    $tokenResult = $tokenStmt->get_result();
    $tokenRow = $tokenResult->fetch_assoc();
    $tokenStmt->close();

    if (!$tokenRow) {
        echo json_encode(['success' => false, 'message' => 'Token invalide. Ce lien de réinitialisation n\'existe pas.']);
        exit;
    }

    // Étape 2: Vérifier si le token est expiré (avec comparaison PHP pour éviter les problèmes de timezone)
    $expiresAt = new DateTime($tokenRow['expires_at']);
    $now = new DateTime();
    
    if ($expiresAt <= $now) {
        echo json_encode(['success' => false, 'message' => 'Token expiré. Veuillez demander un nouveau lien de réinitialisation.']);
        exit;
    }

    // Étape 3: Vérifier si l'utilisateur existe
    $userStmt = $mysqli->prepare('SELECT id, email, username FROM Users WHERE id = ?');
    
    if (!$userStmt) {
        throw new Exception('Erreur de préparation de la requête utilisateur: ' . $mysqli->error);
    }
    
    $userStmt->bind_param('i', $tokenRow['user_id']);
    $userStmt->execute();
    $userResult = $userStmt->get_result();
    $user = $userResult->fetch_assoc();
    $userStmt->close();
    
    if (!$user) {
        echo json_encode(['success' => false, 'message' => 'Token invalide. L\'utilisateur associé n\'existe plus.']);
        exit;
    }

    // Tout est OK, préparer les données
    $tokenData = [
        'user_id' => $user['id'],
        'email' => $user['email'],
        'username' => $user['username'],
        'expires_at' => $tokenRow['expires_at']
    ];

    // Mettre à jour le mot de passe
    $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
    
    if (!$hashedPassword) {
        throw new Exception('Erreur lors du hachage du mot de passe.');
    }
    
    $updateStmt = $mysqli->prepare('UPDATE Users SET password = ? WHERE id = ?');
    
    if (!$updateStmt) {
        throw new Exception('Erreur de préparation de la requête de mise à jour: ' . $mysqli->error);
    }
    
    $updateStmt->bind_param('si', $hashedPassword, $tokenData['user_id']);
    
    if (!$updateStmt->execute()) {
        $updateStmt->close();
        throw new Exception('Erreur lors de la mise à jour du mot de passe: ' . $mysqli->error);
    }
    
    $updateStmt->close();
    
    // Supprimer le token utilisé (et tous les autres tokens de cet utilisateur pour sécurité)
    $deleteStmt = $mysqli->prepare('DELETE FROM password_reset_tokens WHERE user_id = ?');
    $deleteStmt->bind_param('i', $tokenData['user_id']);
    $deleteStmt->execute();
    $deleteStmt->close();
    
    echo json_encode([
        'success' => true, 
        'message' => 'Mot de passe réinitialisé avec succès. Vous pouvez maintenant vous connecter.'
    ]);
    
} catch (Exception $e) {
    error_log('Reset password error: ' . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'Erreur serveur. Veuillez réessayer plus tard.'
    ]);
}
?>
