<?php
header('Content-Type: application/json');
session_start();
require_once '../includes/db.php';

// Définir le fuseau horaire GMT+1
date_default_timezone_set('Etc/GMT-1');
if(!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Non autorisé.']);
    exit;
}
$user_id = $_SESSION['user_id'];
$match_id = intval($_POST['match_id'] ?? 0);
$predicted_team_1 = $_POST['predicted_team_1'] ?? null;
$predicted_team_2 = $_POST['predicted_team_2'] ?? null;
if(!$match_id || $predicted_team_1 === null || $predicted_team_2 === null) {
    echo json_encode(['success' => false, 'message' => 'Données incomplètes.']);
    exit;
}

// Vérifier si c'est une phase d'élimination directe (pas d'égalité autorisée)
$stmt = $mysqli->prepare('SELECT phase, team_1, team_2 FROM Matches WHERE id = ?');
$stmt->bind_param('i', $match_id);
$stmt->execute();
$result = $stmt->get_result();
$match_info = $result->fetch_assoc();
$stmt->close();

if(!$match_info) {
    echo json_encode(['success' => false, 'message' => 'Match introuvable.']);
    exit;
}

// Phases d'élimination directe où l'égalité n'est pas autorisée
$elimination_phases = [
    '8e de finale',
    '8èmes de finale', 
    'Huitièmes de finale',
    'Quart de finale',
    'Quarts de finale',
    'Demi-finale',
    'Demi-finales',
    'Match pour la 3e place',
    '3ème place',
    '3eme place',
    'Finale'
];

$is_elimination_phase = false;
foreach($elimination_phases as $phase) {
    if(strpos($match_info['phase'], $phase) !== false) {
        $is_elimination_phase = true;
        break;
    }
}

// Vérifier l'égalité dans les phases d'élimination directe
if($is_elimination_phase && $predicted_team_1 == $predicted_team_2) {
    echo json_encode([
        'success' => false, 
        'message' => 'Égalité non autorisée',
        'error_type' => 'equality_not_allowed',
        'match_id' => $match_id,
        'current_scores' => [
            'team_1' => $predicted_team_1,
            'team_2' => $predicted_team_2
        ],
        'teams' => [
            'team_1' => $match_info['team_1'],
            'team_2' => $match_info['team_2']
        ]
    ]);
    exit;
}

// Charger la configuration du délai de clôture des pronostics de matchs
$match_deadline_hours_file = __DIR__.'/../includes/match_deadline_hours.txt';
$match_deadline_hours = 1; // Valeur par défaut : 1 heure
if (file_exists($match_deadline_hours_file)) {
    $match_deadline_hours = (int)trim(file_get_contents($match_deadline_hours_file));
    if ($match_deadline_hours <= 0) $match_deadline_hours = 1; // Minimum 1 heure
}

// Récupérer la date du match
$stmt = $mysqli->prepare('SELECT match_date FROM Matches WHERE id = ?');
$stmt->bind_param('i', $match_id);
$stmt->execute();
$result = $stmt->get_result();
$match = $result->fetch_assoc();
$stmt->close();
if(!$match) {
    echo json_encode(['success' => false, 'message' => 'Match introuvable.']);
    exit;
}

// Calculer la deadline : X heures avant le match (en GMT+1)
$deadline = date('Y-m-d H:i:s', strtotime($match['match_date']) - ($match_deadline_hours * 3600));
$current_time = date('Y-m-d H:i:s');

if(date('Y-m-d H:i:s') >= $deadline) {
    echo json_encode(['success' => false, 'message' => 'Pronostic fermé pour ce match.']);
    exit;
}

// Vérifier si un pronostic existe déjà
$stmt = $mysqli->prepare('SELECT id FROM Predictions WHERE user_id = ? AND match_id = ?');
$stmt->bind_param('ii', $user_id, $match_id);
$stmt->execute();
$result = $stmt->get_result();
$existing = $result->fetch_assoc();
$stmt->close();
if($existing) {
    $stmt = $mysqli->prepare('UPDATE Predictions SET predicted_team_1 = ?, predicted_team_2 = ?, prediction_time = NOW() WHERE id = ?');
    $stmt->bind_param('iii', $predicted_team_1, $predicted_team_2, $existing['id']);
    $stmt->execute();
    $stmt->close();
} else {
    $stmt = $mysqli->prepare('INSERT INTO Predictions (user_id, match_id, predicted_team_1, predicted_team_2) VALUES (?, ?, ?, ?)');
    $stmt->bind_param('iiii', $user_id, $match_id, $predicted_team_1, $predicted_team_2);
    $stmt->execute();
    $stmt->close();
}
echo json_encode(['success' => true]); 