<?php
// Inclure le système de langue en premier
require_once 'includes/language.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$user_id = $_SESSION['user_id'] ?? 0;

// Gestion de l'upload AVANT l'inclusion du header
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $user_id) {
    // Connexion DB
    require_once 'includes/db.php';
    
    if (isset($_FILES['photo']) && $_FILES['photo']['error'] === 0) {
        $ext = strtolower(pathinfo($_FILES['photo']['name'], PATHINFO_EXTENSION));
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        if (in_array($ext, $allowed)) {
            // Créer le dossier upload s'il n'existe pas
            $upload_dir = 'assets/img/upload/';
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            $img_path = $upload_dir . 'challenge_' . uniqid() . '.' . $ext;
            if (move_uploaded_file($_FILES['photo']['tmp_name'], $img_path)) {
                // Plus besoin de phase_id, on met null par défaut
                $phase_id = null;
                $stmt = $mysqli->prepare('INSERT INTO challenge_photos (user_id, image_path, phase_id) VALUES (?, ?, ?)');
                if ($stmt) {
                    $stmt->bind_param('isi', $user_id, $img_path, $phase_id);
                    if ($stmt->execute()) {
                        $stmt->close();
                        // Redirection pour éviter la re-soumission du formulaire
                        $_SESSION['upload_success'] = true;
                        header('Location: challenge_photos.php');
                        exit;
                    } else {
                        unlink($img_path); // Supprimer le fichier si la DB échoue
                        $stmt->close();
                        $_SESSION['upload_error'] = 'upload_error';
                        header('Location: challenge_photos.php');
                        exit;
                    }
                } else {
                    unlink($img_path);
                    $_SESSION['upload_error'] = 'upload_error';
                    header('Location: challenge_photos.php');
                    exit;
                }
            } else {
                $_SESSION['upload_error'] = 'upload_error';
                header('Location: challenge_photos.php');
                exit;
            }
        } else {
            $_SESSION['upload_error'] = 'format_not_allowed';
            header('Location: challenge_photos.php');
            exit;
        }
    } else {
        $error_code = $_FILES['photo']['error'] ?? 'unknown';
        $_SESSION['upload_error'] = 'upload_error';
        $_SESSION['upload_error_code'] = $error_code;
        header('Location: challenge_photos.php');
        exit;
    }
}

// Maintenant on peut inclure le header
include 'templates/header.php';

// Connexion DB (si pas déjà fait)
if (!isset($mysqli)) {
    require_once 'includes/db.php';
}

// Affichage des messages après redirection
$msg = '';
if (isset($_SESSION['upload_success'])) {
    $msg = '<span style="color:#4caf50;">' . __('photo_uploaded') . '</span>';
    unset($_SESSION['upload_success']);
} elseif (isset($_SESSION['upload_error'])) {
    $error_type = $_SESSION['upload_error'];
    $msg = '<span style="color:#ff5252;">' . __($error_type) . '</span>';
    unset($_SESSION['upload_error']);
}

// Fonction pour récupérer les phases actives des défis photos
function getActiveChallengePhotosPhases($mysqli) {
    $phases = [];
    $current_lang = getCurrentLanguage();
    
    try {
        // Vérifier si la table phase_management existe
        $result = $mysqli->query("SHOW TABLES LIKE 'phase_management'");
        if ($result && $result->num_rows > 0) {
            // Utiliser la nouvelle table phase_management
            $stmt = $mysqli->prepare("SELECT phase_name, is_active FROM phase_management WHERE page_type = 'challenge_photos' ORDER BY 
                CASE phase_name 
                    WHEN 'Phase de groupes' THEN 1
                    WHEN '8e de finale' THEN 2
                    WHEN 'Quart de finale' THEN 3
                    WHEN 'Demi-finale' THEN 4
                    WHEN 'Finale' THEN 5
                    WHEN 'Match pour la 3e place' THEN 6
                    ELSE 7
                END");
            
            if ($stmt) {
                $stmt->execute();
                $result = $stmt->get_result();
                
                $has_third_place = false;
                $has_finale = false;
                $third_place_data = null;
                $finale_data = null;
                
                while ($row = $result->fetch_assoc()) {
                    // Si c'est "Match pour la 3e place", on ne l'ajoute pas directement mais on note qu'elle existe
                    if ($row['phase_name'] === 'Match pour la 3e place') {
                        $has_third_place = true;
                        // Récupérer l'id, title et description depuis photo_challenge_phases si elle existe
                        $desc_stmt = $mysqli->prepare("SELECT id, title, description FROM photo_challenge_phases WHERE phase = ? LIMIT 1");
                        $desc_stmt->bind_param('s', $row['phase_name']);
                        $desc_stmt->execute();
                        $desc_result = $desc_stmt->get_result();
                        $desc_row = $desc_result->fetch_assoc();
                        $desc_stmt->close();
                        
                        $third_place_data = [
                            'id' => $desc_row['id'] ?? null,
                            'phase' => $row['phase_name'],
                            'is_active' => $row['is_active']
                        ];
                        continue; // Ne pas l'ajouter à la liste des phases affichées
                    }
                    
                    // Si c'est "Finale", on note qu'elle existe
                    if ($row['phase_name'] === 'Finale') {
                        $has_finale = true;
                        // Récupérer l'id, title et description depuis photo_challenge_phases si elle existe
                        $desc_stmt = $mysqli->prepare("SELECT id, title, description FROM photo_challenge_phases WHERE phase = ? LIMIT 1");
                        $desc_stmt->bind_param('s', $row['phase_name']);
                        $desc_stmt->execute();
                        $desc_result = $desc_stmt->get_result();
                        $desc_row = $desc_result->fetch_assoc();
                        $desc_stmt->close();
                        
                        $finale_data = [
                            'id' => $desc_row['id'] ?? null,
                            'phase' => $row['phase_name'],
                            'is_active' => $row['is_active']
                        ];
                        continue; // Ne pas l'ajouter à la liste des phases affichées
                    }
                    
                    // Pour les autres phases, traitement normal
                    // Récupérer l'id, title et description depuis photo_challenge_phases si elle existe
                    $desc_stmt = $mysqli->prepare("SELECT id, title, description FROM photo_challenge_phases WHERE phase = ? LIMIT 1");
                    $desc_stmt->bind_param('s', $row['phase_name']);
                    $desc_stmt->execute();
                    $desc_result = $desc_stmt->get_result();
                    $desc_row = $desc_result->fetch_assoc();
                    $desc_stmt->close();
                    
                    // Récupérer l'id de la phase
                    $phase_id = $desc_row['id'] ?? null;
                    
                    // Déterminer la description avec traduction
                    $description = '';
                    if ($desc_row) {
                        $description = $desc_row['description'] ?: $desc_row['title'] ?: '';
                    }
                    
                    // Si pas de description, utiliser un texte par défaut traduit
                    if (empty($description)) {
                        $description = 'Défi photo pour ' . $row['phase_name'];
                    }
                    
                    // Traduire le texte spécifique "Partagez vos photos de la phase de groupes de la CAN 2025"
                    if (stripos($description, 'Partagez vos photos de la phase de groupes de la CAN 2025') !== false) {
                        $description = __('share_photos_group_phase');
                    } elseif (stripos($description, 'Défi photo pour') !== false) {
                        // Traduire le texte de fallback
                        $phase_translated = '';
                        switch($row['phase_name']) {
                            case 'Phase de groupes':
                                $phase_translated = __('group_phase');
                                break;
                            case '8e de finale':
                                $phase_translated = __('round_of_16');
                                break;
                            case 'Quart de finale':
                                $phase_translated = __('quarter_final');
                                break;
                            case 'Demi-finale':
                                $phase_translated = __('semi_final');
                                break;
                            default:
                                $phase_translated = $row['phase_name'];
                        }
                        $description = ($current_lang === 'ar') ? 'تحدي الصور لـ ' . $phase_translated : 'Défi photo pour ' . $phase_translated;
                    }
                    
                    $phases[] = [
                        'id' => $phase_id,
                        'phase' => $row['phase_name'],
                        'description' => $description,
                        'is_active' => $row['is_active']
                    ];
                }
                
                // Vérifier spécifiquement si "Finale" est active pour la page challenge_photos
                // L'onglet "Finale" ne sera affiché que si "Finale" est active
                $finale_is_active = false;
                $stmt_finale_check = $mysqli->prepare("SELECT is_active FROM phase_management WHERE page_type = 'challenge_photos' AND phase_name = 'Finale'");
                if ($stmt_finale_check) {
                    $stmt_finale_check->execute();
                    $result_finale_check = $stmt_finale_check->get_result();
                    $row_finale_check = $result_finale_check->fetch_assoc();
                    $stmt_finale_check->close();
                    
                    if ($row_finale_check && $row_finale_check['is_active'] == 1) {
                        $finale_is_active = true;
                    }
                }
                
                // Combiner Match pour la 3e place et Finale en un seul onglet "Finale"
                // MAIS seulement si "Finale" est active
                if ($finale_is_active) {
                    $finale_translated = __('final');
                    $combined_description = ($current_lang === 'ar') 
                        ? 'تحدي الصور لـ ' . $finale_translated
                        : 'Défi photo pour ' . $finale_translated;
                    
                    // Récupérer la description de la Finale si elle existe
                    if ($has_finale && $finale_data) {
                        $desc_stmt = $mysqli->prepare("SELECT title, description FROM photo_challenge_phases WHERE phase = 'Finale' LIMIT 1");
                        $desc_stmt->execute();
                        $desc_result = $desc_stmt->get_result();
                        $desc_row = $desc_result->fetch_assoc();
                        $desc_stmt->close();
                        
                        if ($desc_row && !empty($desc_row['description'])) {
                            $combined_description = $desc_row['description'];
                        } elseif ($desc_row && !empty($desc_row['title'])) {
                            $combined_description = $desc_row['title'];
                        }
                    }
                    
                    $phases[] = [
                        'id' => $finale_data['id'] ?? $third_place_data['id'] ?? null,
                        'phase' => 'Finale',
                        'description' => $combined_description,
                        'is_active' => true,
                        'combined_phases' => ['Finale', 'Match pour la 3e place'] // Pour le filtrage des photos
                    ];
                }
                
                $stmt->close();
            }
        } else {
            // Fallback vers l'ancienne table photo_challenge_phases
            $res_phases = $mysqli->query("SELECT * FROM photo_challenge_phases WHERE is_active = 1 ORDER BY id");
            if ($res_phases) {
                $phases_data = $res_phases->fetch_all(MYSQLI_ASSOC);
                $res_phases->close();
                
                // Traduire les descriptions
                foreach ($phases_data as &$phase_data) {
                    if (isset($phase_data['description'])) {
                        if (stripos($phase_data['description'], 'Partagez vos photos de la phase de groupes de la CAN 2025') !== false) {
                            $phase_data['description'] = __('share_photos_group_phase');
                        }
                    }
                }
                $phases = $phases_data;
            }
        }
    } catch (Exception $e) {
        // En cas d'erreur, utiliser l'ancienne table
        $res_phases = $mysqli->query("SELECT * FROM photo_challenge_phases WHERE is_active = 1 ORDER BY id");
        if ($res_phases) {
            $phases_data = $res_phases->fetch_all(MYSQLI_ASSOC);
            $res_phases->close();
            
            // Traduire les descriptions
            foreach ($phases_data as &$phase_data) {
                if (isset($phase_data['description'])) {
                    if (stripos($phase_data['description'], 'Partagez vos photos de la phase de groupes de la CAN 2025') !== false) {
                        $phase_data['description'] = __('share_photos_group_phase');
                    }
                }
            }
            $phases = $phases_data;
        }
    }
    
    return $phases;
}

// Fonction helper pour traduire le nom de la phase
function translatePhaseName($phase_name) {
    switch($phase_name) {
        case 'Phase de groupes':
            return __('group_phase');
        case '8e de finale':
            return __('round_of_16');
        case 'Quart de finale':
            return __('quarter_final');
        case 'Demi-finale':
            return __('semi_final');
        case 'Match pour la 3e place':
            return __('third_place_match');
        case 'Finale':
            return __('final');
        default:
            return $phase_name;
    }
}

// Récupérer la langue courante
$current_lang = getCurrentLanguage();

// Récupérer les phases actives
$phases = getActiveChallengePhotosPhases($mysqli);

// Récupérer la phase sélectionnée
$selected_phase = $_GET['phase'] ?? '';

// Affichage des images avec filtrage par phase
$where_clause = '';
$params = [];
$param_types = '';

if ($selected_phase) {
    $where_clause = ' WHERE ph.phase = ?';
    $params[] = $selected_phase;
    $param_types = 's';
}

$query = 'SELECT p.*, u.username, ph.phase, ph.title as phase_title, 
          (SELECT COUNT(*) FROM challenge_likes l WHERE l.photo_id = p.id) as likes, 
          (SELECT COUNT(*) FROM challenge_likes l WHERE l.photo_id = p.id AND l.user_id = ?) as liked 
          FROM challenge_photos p 
          LEFT JOIN Users u ON p.user_id = u.id 
          LEFT JOIN photo_challenge_phases ph ON p.phase_id = ph.id' . 
          $where_clause . ' 
          ORDER BY likes DESC, p.created_at DESC';

$params[] = intval($user_id);
$param_types .= 'i';

$stmt = $mysqli->prepare($query);
if (!$stmt) {
    // If the query fails, it might be because tables don't exist yet
    // Let's create a simple fallback query
    $fallback_query = 'SELECT p.*, u.username, NULL as phase, NULL as phase_title, 0 as likes, 0 as liked 
                       FROM challenge_photos p 
                       LEFT JOIN Users u ON p.user_id = u.id 
                       ORDER BY likes DESC, p.created_at DESC';
    $stmt = $mysqli->prepare($fallback_query);
    if (!$stmt) {
        // If even the fallback fails, return empty array
        $photos = [];
    } else {
        $stmt->execute();
        $result = $stmt->get_result();
        $photos = $result->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }
} else {
    if (!empty($params)) {
        $stmt->bind_param($param_types, ...$params);
    }
    $stmt->execute();
    $result = $stmt->get_result();
    $photos = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}
?>
<style>
.mur-container {
  max-width: 900px;
  margin: 40px auto 30px auto;
  background: #23232b;
  border-radius: 18px;
  box-shadow: 0 4px 24px #0005;
  padding: 32px 24px 24px 24px;
  min-height: auto;
  overflow: visible;
}

.mur-container.page-challenge-photos {
  overflow: visible;
}
.mur-title {
  color: #ffb300;
  font-size: 2rem;
  font-weight: 800;
  text-align: center;
  margin-bottom: 18px;
  letter-spacing: 1px;
}
#challenge-form {
  display: flex;
  flex-direction: column;
  gap: 10px;
  margin-bottom: 18px;
}
#challenge-form input[type=file] {
  background: #fffbe6;
  border-radius: 8px;
  padding: 6px;
}
#challenge-form button {
  background: linear-gradient(90deg, #ffb300 60%, #fffbe6 100%);
  color: #23232b;
  border: none;
  border-radius: 10px;
  font-weight: 800;
  font-size: 1.13rem;
  padding: 10px 28px;
  margin-top: 6px;
  cursor: pointer;
}
.mur-message-card {
  background: #29293a;
  border-radius: 14px;
  margin-bottom: 18px;
  padding: 14px 12px;
  box-shadow: 0 2px 8px #0002;
  border: 2px solid #ffb30044;
  display: flex;
  flex-direction: column;
  gap: 8px;
  position: relative;
  color: white;
}
.mur-message-img {
  max-width: 100%;
  max-height: 500px;
  width: auto;
  height: auto;
  object-fit: contain;
  border-radius: 8px;
  margin-top: 6px;
}
.mur-message-footer {
  display: flex;
  align-items: center;
  gap: 12px;
  font-size: 0.98rem;
}
.mur-like-btn {
  background: none;
  border: none;
  color: #ffb300;
  font-size: 1.3rem;
  cursor: pointer;
  display: flex;
  align-items: center;
  gap: 4px;
}
.mur-like-btn.liked {
  color: #e53935;
}
.mur-message-pseudo {
  color: #ffb300;
  font-weight: 700;
  font-size: 1.08rem;
  margin-bottom: 2px;
}

/* Styles pour le bouton flottant */
.mur-fab {
  position: fixed;
  right: 32px;
  bottom: 32px;
  width: 56px;
  height: 56px;
  border-radius: 50%;
  background: linear-gradient(90deg, #ffffff 60%, #fffbe6 100%);
  color: #23232b;
  font-size: 2.2rem;
  font-weight: 900;
  border: none;
  box-shadow: 0 4px 24px #0005;
  cursor: pointer;
  z-index: 1001;
  transition: transform 0.12s;
  display: flex;
  align-items: center;
  justify-content: center;
  overflow: hidden;
}

.mur-fab:hover {
  transform: scale(1.08);
}

/* Styles pour les phases */
.phase-section {
  background: rgba(255, 255, 255, 0.05);
  border-radius: 15px;
  padding: 25px;
  margin-bottom: 30px;
  border: 1px solid rgba(255, 255, 255, 0.1);
}

.phase-header {
  text-align: center;
  margin-bottom: 25px;
  padding-bottom: 20px;
  border-bottom: 2px solid rgba(255, 179, 0, 0.3);
}

.phase-title {
  color: #ffb300;
  font-size: 1.8rem;
  font-weight: 700;
  margin-bottom: 10px;
  text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
}

.phase-description {
  color: #ffffff;
  font-size: 1.1rem;
  opacity: 0.9;
  margin: 0;
}

.no-photos-message {
  text-align: center;
  color: #ffb300;
  padding: 40px 20px;
  background: rgba(255, 179, 0, 0.1);
  border-radius: 10px;
  border: 2px dashed rgba(255, 179, 0, 0.3);
}

.no-photos-message i {
  color: #ffb300;
  opacity: 0.7;
}

.no-photos-message p {
  margin: 10px 0;
  font-weight: 600;
}

.photos-grid {
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
  gap: 20px;
  margin-top: 20px;
}

/* Styles pour les tabs */
.challenge-tabs {
  border-bottom: 2px solid rgba(255, 179, 0, 0.3);
}

.challenge-tabs .nav-link {
  color: #ffffff;
  background: transparent;
  border: none;
  border-radius: 8px 8px 0 0;
  padding: 12px 20px;
  font-weight: 600;
  transition: all 0.3s ease;
}

.challenge-tabs .nav-link:hover {
  color: #ffb300;
  background: rgba(255, 179, 0, 0.1);
}

.challenge-tabs .nav-link.active {
  color: #ffb300;
  background: rgba(255, 179, 0, 0.2);
  border-bottom: 2px solid #ffb300;
}

.tab-content {
  background: rgba(255, 255, 255, 0.05);
  border-radius: 0 0 15px 15px;
  padding: 25px;
  min-height: 400px;
  overflow-y: visible;
  scrollbar-width: thin;
  scrollbar-color: #ffb300 #2a2a3a;
}

@media (max-width: 768px) {
  .photos-grid {
    grid-template-columns: 1fr;
  }
  
  .phase-title {
    font-size: 1.5rem;
  }
  
  .phase-section {
    padding: 20px 15px;
  }
  
  .challenge-tabs .nav-link {
    padding: 8px 12px;
    font-size: 0.9rem;
  }
  
  .tab-content {
    padding: 15px;
  }
  
  .mur-message-img {
    max-height: 350px;
  }
}

.mur-modal-bg {
  position: fixed;
  top: 0; left: 0; right: 0; bottom: 0;
  background: rgba(30,30,40,0.85);
  z-index: 1000;
  display: flex;
  align-items: center;
  justify-content: center;
}

.mur-modal {
  background: #23232b;
  border-radius: 18px;
  box-shadow: 0 4px 24px #000a;
  padding: 32px 28px 28px 28px;
  min-width: 600px;
  max-width: 750px;
  position: relative;
}

.mur-modal-close {
  position: absolute;
  top: 12px; right: 16px;
  background: none;
  border: none;
  color: #ffb300;
  font-size: 2rem;
  font-weight: 900;
  cursor: pointer;
}

/* Styles RTL pour l'arabe */
.lang-ar .mur-fab {
  right: auto;
  left: 32px;
}

.lang-ar .mur-modal-close {
  right: auto;
  left: 16px;
}

@media (min-width: 768px) and (max-width: 1023.98px) {
  .mur-container {
    max-width: 700px;
  }
}

@media (max-width: 600px) {
  .mur-container { padding: 8px 2vw; }
  .mur-title { font-size: 1.3rem; }
  #challenge-form input[type=file] { font-size: 1rem; }
  .mur-modal { min-width: 0; max-width: 95vw; padding: 24px 16px 20px 16px; }
}
</style>

<!-- Script LottieFiles -->
<script src="https://unpkg.com/@lottiefiles/dotlottie-wc@0.6.2/dist/dotlottie-wc.js" type="module"></script>

<div class="mur-container page-challenge-photos">
  <div class="mur-title"><?= __('challenge_title') ?></div>
  
  <!-- Texte du challenge -->
  <div class="challenge-text mb-4" style="text-align: center; color: #ffffff; padding: 20px; background: rgba(255, 179, 0, 0.1); border-radius: 12px; border: 2px solid rgba(255, 179, 0, 0.3);">
    <h2 style="color: #ffb300; font-size: 1.5rem; font-weight: 700; margin-bottom: 15px;">#BMCItogether</h2>
    <p style="font-size: 1.1rem; line-height: 1.6; margin-bottom: 10px;">
      <?= ($current_lang === 'ar') 
        ? 'شارك لحظات قوية أمام مباراة من كأس أمم إفريقيا مع زملائك واحفظ هذه اللحظة!' 
        : 'Partagez des moments forts devant un match de la CAN avec vos collègues et immortalisez ce moment!' ?>
    </p>
    <p style="font-size: 1.1rem; line-height: 1.6; margin-bottom: 10px; color: #ffb300; font-weight: 600;">
      <?= ($current_lang === 'ar') 
        ? 'الصورة التي تجمع أكبر عدد من زملاء BMCI تفوز بالمسابقة!' 
        : 'La photo qui regroupe le plus de collègues BMCI remporte le concours!' ?>
    </p>
    <p style="font-size: 1rem; line-height: 1.6; color: rgba(255, 255, 255, 0.9);">
      <?= ($current_lang === 'ar') 
        ? 'للربح: قسيمة شراء بقيمة 500 درهم من متجر أدوات الرياضة لكل موظف(ة) موجود(ة) في الصورة!' 
        : 'À gagner, un bon d\'achat de 500 dhs chez un magasin d\'articles de sport pour chaque collaborateur(rice) présent(e) sur la photo!' ?>
    </p>
  </div>
  
  <!-- Affichage de toutes les photos sans filtrage par phase -->
  <?php 
  // Trier toutes les photos par nombre de likes décroissant
  usort($photos, function($a, $b) {
    if ($b['likes'] === $a['likes']) {
      return strtotime($b['created_at']) - strtotime($a['created_at']); // Si même nombre de likes, trier par date
    }
    return $b['likes'] - $a['likes'];
  });
  ?>
  
  <?php if (empty($photos)): ?>
    <div class="no-photos-message">
      <i class="fas fa-camera fa-3x mb-3"></i>
      <p><?= __('no_photos') ?></p>
      <p class="text-muted"><?= __('be_first_to_participate') ?></p>
    </div>
  <?php else: ?>
    <div class="photos-grid">
      <?php foreach ($photos as $photo): ?>
        <div class="mur-message-card">
          <div class="mur-message-pseudo">
            <?= htmlspecialchars($photo['username'] ?? __('user')) ?>
          </div>
          <img src="<?= htmlspecialchars($photo['image_path']) ?>" class="mur-message-img" alt="photo challenge" />
          <div class="mur-message-footer">
            <form method="post" action="like_challenge_photo.php" style="display:inline;">
              <input type="hidden" name="photo_id" value="<?= $photo['id'] ?>" />
              <button type="submit" class="mur-like-btn<?= $photo['liked'] ? ' liked' : '' ?>" name="like">
                <span><?= $photo['liked'] ? '❤️' : '🤍' ?></span> <span><?= $photo['likes'] ?></span>
              </button>
            </form>
          </div>
        </div>
      <?php endforeach; ?>
    </div>
  <?php endif; ?>
</div>

<!-- Bouton flottant avec animation Lottie -->
<dotlottie-wc
  id="openPhotoPopup"
  class="mur-fab"
  src="https://lottie.host/75265b49-3815-4282-a3d1-9f635943bc56/qPOOBhabqB.lottie"
  style="width: 56px; height: 56px; cursor: pointer;"
  speed="1"
  autoplay
  loop
  title="<?= __('upload_photo') ?>"
  onerror="this.style.display='none'; document.getElementById('openPhotoPopupFallback').style.display='flex';"
></dotlottie-wc>

<!-- Bouton fallback si l'animation ne charge pas -->
<button
  id="openPhotoPopupFallback"
  class="mur-fab"
  style="display: none;"
  title="<?= __('upload_photo') ?>"
>
  +
</button>

<!-- Popup modal pour upload photo -->
<div id="photoModal" class="mur-modal-bg" style="display:none;">
  <div class="mur-modal">
    <button class="mur-modal-close" id="closePhotoPopup">&times;</button>
    <?php if ($user_id): ?>
      <form id="challenge-form" method="post" enctype="multipart/form-data">
        <h3 style="color: #ffb300; margin-bottom: 15px;"><?= __('upload_photo') ?></h3>
        
        
        <input type="file" name="photo" accept="image/*" required />
        <button type="submit"><?= __('upload') ?></button>
        <div style="margin-top:6px;min-height:22px;"><?= $msg ?></div>
      </form>
    <?php else: ?>
      <div style="color:#ffb300;text-align:center;margin-bottom:24px;"><?= __('login_to_participate') ?></div>
    <?php endif; ?>
  </div>
</div>

<script>
function filterByPhase(phase) {
  const url = new URL(window.location);
  if (phase) {
    url.searchParams.set('phase', phase);
  } else {
    url.searchParams.delete('phase');
  }
  window.location.href = url.toString();
}

window.addEventListener('DOMContentLoaded', function() {
  var openBtn = document.getElementById('openPhotoPopup');
  var openBtnFallback = document.getElementById('openPhotoPopupFallback');
  var closeBtn = document.getElementById('closePhotoPopup');
  var photoModal = document.getElementById('photoModal');
  
  // Gestion du bouton principal (animation Lottie)
  if (openBtn && photoModal) {
    openBtn.addEventListener('click', function(e) {
      e.preventDefault();
      photoModal.style.display = 'flex';
    });
  }
  
  // Gestion du bouton fallback
  if (openBtnFallback && photoModal) {
    openBtnFallback.addEventListener('click', function(e) {
      e.preventDefault();
      photoModal.style.display = 'flex';
    });
  }
  
  if (closeBtn && photoModal) {
    closeBtn.addEventListener('click', function(e) {
      e.preventDefault();
      photoModal.style.display = 'none';
    });
  }
  
  if (photoModal) {
    photoModal.addEventListener('click', function(e) {
      if (e.target === this) this.style.display = 'none';
    });
  }

  // Compression image avant upload
  var form = document.getElementById('challenge-form');
  if (form) {
    var fileInput = form.querySelector('input[type="file"]');
    fileInput.addEventListener('change', function(e) {
      const file = e.target.files[0];
      if (!file) return;
      const reader = new FileReader();
      reader.onload = function(event) {
        const img = new Image();
        img.onload = function() {
          const canvas = document.createElement('canvas');
          const maxDim = 800;
          let w = img.width, h = img.height;
          if (w > h && w > maxDim) { h *= maxDim / w; w = maxDim; }
          else if (h > maxDim) { w *= maxDim / h; h = maxDim; }
          canvas.width = w; canvas.height = h;
          canvas.getContext('2d').drawImage(img, 0, 0, w, h);
          canvas.toBlob(function(blob) {
            const newFile = new File([blob], file.name, {type: 'image/jpeg'});
            const dt = new DataTransfer();
            dt.items.add(newFile);
            fileInput.files = dt.files;
          }, 'image/jpeg', 0.7);
        };
        img.src = event.target.result;
      };
      reader.readAsDataURL(file);
    });
  }

  // Gestion AJAX des likes/dislikes
  document.querySelectorAll('.mur-message-footer form').forEach(function(form) {
    form.addEventListener('submit', async function(e) {
      e.preventDefault();
      const btn = form.querySelector('button');
      const photoId = form.querySelector('input[name="photo_id"]').value;
      btn.disabled = true;
      const res = await fetch('like_challenge_photo.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'photo_id=' + encodeURIComponent(photoId)
      });
      if (res.ok) {
        // Mettre à jour le cœur et le compteur sans reload
        const heartSpan = btn.querySelector('span:first-child');
        const countSpan = btn.querySelector('span:last-child');
        const isLiked = btn.classList.contains('liked');
        let count = parseInt(countSpan.textContent, 10) || 0;
        if (isLiked) {
          btn.classList.remove('liked');
          heartSpan.textContent = '🤍';
          countSpan.textContent = count - 1;
        } else {
          btn.classList.add('liked');
          heartSpan.textContent = '❤️';
          countSpan.textContent = count + 1;
        }
      }
      btn.disabled = false;
    });
  });
 });
 </script>
 <?php include 'templates/footer.php'; ?> 