# Système de Traduction CAN 2025

## Vue d'ensemble

Le système de traduction permet de gérer le contenu en français et en arabe pour l'application CAN 2025. Il inclut :

- Gestion automatique des langues (français/arabe)
- Images adaptées selon la langue
- Support RTL pour l'arabe
- Sélecteur de langue avec drapeaux
- Traductions centralisées

## Structure des fichiers

```
lang/
├── fr.php          # Traductions françaises
└── ar.php          # Traductions arabes

includes/
└── language.php    # Système de gestion des langues

ajax/
└── change_language.php  # API pour changer de langue

assets/img/
├── AR mobile/      # Images arabes pour mobile
├── AR pc/          # Images arabes pour desktop
├── icon_menu_mobile/    # Images françaises pour mobile
├── icon_menu_pc/        # Images françaises pour desktop
└── flags/          # Drapeaux des langues
    ├── fr.svg
    └── ma.svg
```

## Utilisation

### 1. Inclure le système de langue

```php
require_once 'includes/language.php';
```

### 2. Utiliser les traductions

```php
// Traduction simple
echo __('welcome');

// Dans les templates
<?= __('welcome') ?>
```

### 3. Obtenir la langue actuelle

```php
$current_lang = getCurrentLanguage();
```

### 4. Obtenir les images selon la langue

```php
$image_path = getImagePath('champion can.png', 'menu');
```

### 5. Gérer la direction du texte

```php
$direction = getTextDirection(); // 'rtl' ou 'ltr'
```

## Ajouter de nouvelles traductions

### 1. Ajouter une nouvelle clé dans `lang/fr.php`

```php
'new_key' => 'Nouvelle traduction',
```

### 2. Ajouter la traduction arabe dans `lang/ar.php`

```php
'new_key' => 'ترجمة جديدة',
```

### 3. Utiliser dans le code

```php
echo __('new_key');
```

## Images adaptées selon la langue

Le système utilise automatiquement les bonnes images selon la langue :

- **Français** : `assets/img/icon_menu_pc/` et `assets/img/icon_menu_mobile/`
- **Arabe** : `assets/img/AR pc/` et `assets/img/AR mobile/`

## Support RTL

Le système gère automatiquement la direction du texte :

- **Français** : LTR (left-to-right)
- **Arabe** : RTL (right-to-left)

## Sélecteur de langue

Le sélecteur de langue est intégré dans le header et permet de :

- Changer de langue en temps réel
- Afficher le drapeau de la langue actuelle
- Recharger la page avec la nouvelle langue

## Fonctions disponibles

### `getCurrentLanguage()`
Retourne la langue actuelle ('fr' ou 'ar')

### `setLanguage($lang)`
Change la langue (stockée en session)

### `__($key)`
Retourne la traduction pour la clé donnée

### `getImagePath($image_name, $type)`
Retourne le chemin de l'image adaptée à la langue

### `getTextDirection()`
Retourne la direction du texte ('rtl' ou 'ltr')

### `getLanguageClass()`
Retourne la classe CSS pour la langue actuelle

## Exemple d'utilisation complète

```php
<?php
require_once 'includes/language.php';
include 'templates/header.php';
?>

<div class="container">
    <h1><?= __('welcome_title') ?></h1>
    
    <div class="menu-grid">
        <a href="pronostics.php" class="menu-btn">
            <img src="<?= getImagePath('pronostics.png', 'menu') ?>" 
                 alt="<?= __('match_predictions') ?>">
        </a>
    </div>
    
    <p><?= __('loading') ?></p>
</div>
```

## Personnalisation

### Ajouter une nouvelle langue

1. Créer le fichier `lang/xx.php`
2. Ajouter la langue dans `$available_languages` dans `includes/language.php`
3. Ajouter les images correspondantes dans `assets/img/`

### Modifier le style du sélecteur

Les styles sont dans `assets/style.css` dans la section "Styles pour le système de traduction".

## Notes importantes

- Les traductions sont mises en cache pour les performances
- La langue est stockée en session
- Le système détecte automatiquement la langue du navigateur
- Les images sont adaptées selon le device (mobile/desktop) et la langue