<?php
// Inclure le système de langue en premier
require_once 'includes/language.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if(!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}
include 'templates/header.php';
require_once 'includes/db.php';

// Fonction pour calculer le score d'un utilisateur pour une phase donnée (avec groupes + quiz de la phase)
function getUserScoreForPhase($mysqli, $user_id, $phase) {
    $total_score = 0;
    
    // 1. Points matchs + titulaires pour cette phase spécifique
    if ($phase === "Phase de groupes") {
        // Pour la phase de groupes, calculer le total des 3 journées
        for ($j = 1; $j <= 3; $j++) {
            $phase_journee = "Phase de groupes · Journée $j sur 3";
            $stmt = $mysqli->prepare('SELECT SUM(mp.points + IFNULL(mp.titulaire_11,0)) as total FROM match_points mp JOIN Matches m ON m.id = mp.match_id WHERE mp.user_id = ? AND m.phase = ?');
            $stmt->bind_param('is', $user_id, $phase_journee);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            $stmt->close();
            $total_score += intval($row['total'] ?? 0);
        }
    } elseif ($phase === "Match pour la 3e place & Finale") {
        // Pour la phase combinée, calculer le total des deux phases
        $stmt = $mysqli->prepare('SELECT SUM(mp.points + IFNULL(mp.titulaire_11,0)) as total FROM match_points mp JOIN Matches m ON m.id = mp.match_id WHERE mp.user_id = ? AND (m.phase = ? OR m.phase = ?)');
        $phase1 = 'Match pour la 3e place';
        $phase2 = 'Finale';
        $stmt->bind_param('iss', $user_id, $phase1, $phase2);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();
        $total_score += intval($row['total'] ?? 0);
    } else {
        // Pour les autres phases, calculer normalement
        $stmt = $mysqli->prepare('SELECT SUM(mp.points + IFNULL(mp.titulaire_11,0)) as total FROM match_points mp JOIN Matches m ON m.id = mp.match_id WHERE mp.user_id = ? AND m.phase = ?');
        $stmt->bind_param('is', $user_id, $phase);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();
        $total_score += intval($row['total'] ?? 0);
    }
    
    // 2. Points groupes (uniquement pour la phase de groupes)
    if ($phase === "Phase de groupes") {
        $stmt = $mysqli->prepare('SELECT SUM(points) as total FROM groupe_points WHERE user_id = ?');
        if ($stmt) {
            $stmt->bind_param('i', $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            $stmt->close();
            $total_score += intval($row['total'] ?? 0);
        }
    }
    
    // 3. Points quiz pour cette phase spécifique
    // IMPORTANT: "Match pour la 3e place" n'est plus une phase de quiz séparée, elle est combinée avec "Finale"
    if ($phase === "Match pour la 3e place & Finale") {
        // Pour la phase combinée, chercher uniquement "Finale" (car les quiz sont maintenant combinés)
        // Mais aussi vérifier les anciens scores de "Match pour la 3e place" pour compatibilité
        $stmt = $mysqli->prepare('SELECT SUM(score) as total FROM quizz_results_phases WHERE user_id = ? AND (phase = ? OR phase = ?)');
        $phase1 = 'Match pour la 3e place';
        $phase2 = 'Finale';
        $stmt->bind_param('iss', $user_id, $phase1, $phase2);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();
        
        // Si les deux phases ont des scores, prendre seulement "Finale" (le plus récent)
        // Sinon, prendre celui qui existe
        $phase_finale = 'Finale';
        $stmt_finale = $mysqli->prepare('SELECT score FROM quizz_results_phases WHERE user_id = ? AND phase = ?');
        $stmt_finale->bind_param('is', $user_id, $phase_finale);
        $stmt_finale->execute();
        $result_finale = $stmt_finale->get_result();
        $row_finale = $result_finale->fetch_assoc();
        $stmt_finale->close();
        
        if ($row_finale && $row_finale['score']) {
            $total_score += intval($row_finale['score'] ?? 0);
        } else {
            // Si "Finale" n'a pas de score, utiliser "Match pour la 3e place" (ancien système)
            $phase_third = 'Match pour la 3e place';
            $stmt_third = $mysqli->prepare('SELECT score FROM quizz_results_phases WHERE user_id = ? AND phase = ?');
            $stmt_third->bind_param('is', $user_id, $phase_third);
            $stmt_third->execute();
            $result_third = $stmt_third->get_result();
            $row_third = $result_third->fetch_assoc();
            $stmt_third->close();
            $total_score += intval($row_third['score'] ?? 0);
        }
    } else {
        $stmt = $mysqli->prepare('SELECT score FROM quizz_results_phases WHERE user_id = ? AND phase = ?');
        if ($stmt) {
            $stmt->bind_param('is', $user_id, $phase);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            $stmt->close();
            $total_score += intval($row['score'] ?? 0);
        }
    }
    
    return $total_score;
}

// Fonction pour calculer le score total d'un utilisateur (toutes phases + bonus)
function getUserTotalScore($mysqli, $user_id) {
    $total_score = 0;
    
    // 1. Points matchs + titulaires pour toutes les phases
    $stmt = $mysqli->prepare('SELECT SUM(mp.points + IFNULL(mp.titulaire_11,0)) as total FROM match_points mp WHERE mp.user_id = ?');
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $stmt->close();
    $total_score += intval($row['total'] ?? 0);
    
    // 2. Points groupes
    $stmt = $mysqli->prepare('SELECT SUM(points) as total FROM groupe_points WHERE user_id = ?');
    if ($stmt) {
        $stmt->bind_param('i', $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();
        $total_score += intval($row['total'] ?? 0);
    }
    
    // 3. Points vainqueur CAN
    $winner_team = null;
    $result = $mysqli->query('SELECT team FROM winner_result LIMIT 1');
    if ($row = $result->fetch_assoc()) {
        $winner_team = $row['team'];
    }
    $result->close();
    
    if ($winner_team) {
        $stmt = $mysqli->prepare("SELECT team FROM winner_predictions WHERE user_id = ?");
        $stmt->bind_param('i', $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();
        
        $points_vainqueur_can = file_exists('includes/points_vainqueur_can.txt') ? intval(trim(file_get_contents('includes/points_vainqueur_can.txt'))) : 100;
        if ($row && $row['team'] === $winner_team) {
            $total_score += $points_vainqueur_can;
        }
    }
    
    // 4. Points quiz pour toutes les phases (nouveau système)
    // IMPORTANT: "Match pour la 3e place" n'est plus une phase de quiz séparée, elle est combinée avec "Finale"
    // Récupérer tous les scores de quiz sauf "Match pour la 3e place" (car combiné avec "Finale")
    $stmt = $mysqli->prepare("SELECT SUM(score) as total FROM quizz_results_phases WHERE user_id = ? AND phase != ?");
    $exclude_phase = 'Match pour la 3e place';
    $stmt->bind_param('is', $user_id, $exclude_phase);
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();
        $total_score += intval($row['total'] ?? 0);
    }
    
    // Si "Finale" n'a pas de score mais "Match pour la 3e place" en a un, l'inclure (compatibilité)
    $phase_finale = 'Finale';
    $stmt_check_finale = $mysqli->prepare('SELECT score FROM quizz_results_phases WHERE user_id = ? AND phase = ?');
    $stmt_check_finale->bind_param('is', $user_id, $phase_finale);
    $stmt_check_finale->execute();
    $result_check_finale = $stmt_check_finale->get_result();
    $row_check_finale = $result_check_finale->fetch_assoc();
    $stmt_check_finale->close();
    
    if (!$row_check_finale || !$row_check_finale['score']) {
        // Si "Finale" n'a pas de score, vérifier "Match pour la 3e place"
        $phase_third = 'Match pour la 3e place';
        $stmt_third = $mysqli->prepare('SELECT score FROM quizz_results_phases WHERE user_id = ? AND phase = ?');
        $stmt_third->bind_param('is', $user_id, $phase_third);
        $stmt_third->execute();
        $result_third = $stmt_third->get_result();
        $row_third = $result_third->fetch_assoc();
        $stmt_third->close();
        
        if ($row_third && $row_third['score']) {
            $total_score += intval($row_third['score'] ?? 0);
        }
    }
    
    return $total_score;
}

// Fonction pour formater le temps en millisecondes en format lisible (mm:ss:mmm)
function formatQuizTime($milliseconds) {
    if ($milliseconds === PHP_INT_MAX || $milliseconds === null) {
        return '-';
    }
    $total_seconds = floor($milliseconds / 1000);
    $minutes = floor($total_seconds / 60);
    $secs = $total_seconds % 60;
    $millis = $milliseconds % 1000;
    return sprintf('%02d:%02d:%03d', $minutes, $secs, $millis);
}

// Fonction pour obtenir le temps passé dans le quiz pour une phase (en millisecondes)
function getQuizTimeForPhase($mysqli, $user_id, $phase_name) {
    // Gérer le cas de la phase combinée "Match pour la 3e place & Finale"
    if ($phase_name === "Match pour la 3e place & Finale") {
        // Récupérer le temps pour "Finale" (priorité) ou "Match pour la 3e place"
        // Utiliser TIMESTAMPDIFF avec MICROSECOND pour obtenir les microsecondes, puis convertir en millisecondes
        $stmt = $mysqli->prepare('SELECT TIMESTAMPDIFF(MICROSECOND, start_time, completed_at) as time_diff FROM quizz_results_phases WHERE user_id = ? AND phase = ? AND start_time IS NOT NULL AND completed_at IS NOT NULL');
        $phase_finale = 'Finale';
        $stmt->bind_param('is', $user_id, $phase_finale);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();
        
        // Si pas de résultat pour Finale, chercher Match pour la 3e place
        if (!$row || !$row['time_diff']) {
            $phase_third = 'Match pour la 3e place';
            $stmt = $mysqli->prepare('SELECT TIMESTAMPDIFF(MICROSECOND, start_time, completed_at) as time_diff FROM quizz_results_phases WHERE user_id = ? AND phase = ? AND start_time IS NOT NULL AND completed_at IS NOT NULL');
            $stmt->bind_param('is', $user_id, $phase_third);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            $stmt->close();
        }
    } else {
        // Pour les autres phases, récupérer directement avec TIMESTAMPDIFF
        $stmt = $mysqli->prepare('SELECT TIMESTAMPDIFF(MICROSECOND, start_time, completed_at) as time_diff FROM quizz_results_phases WHERE user_id = ? AND phase = ? AND start_time IS NOT NULL AND completed_at IS NOT NULL');
        $stmt->bind_param('is', $user_id, $phase_name);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();
    }
    
    // Calculer le temps passé en millisecondes (convertir les microsecondes en millisecondes)
    if ($row && isset($row['time_diff']) && $row['time_diff'] !== null) {
        // Convertir les microsecondes en millisecondes (diviser par 1000)
        $milliseconds = intval($row['time_diff'] / 1000);
        return $milliseconds;
    }
    
    // Si pas de temps disponible, retourner une valeur très élevée pour être en dernier
    return PHP_INT_MAX;
}

// Fonction pour obtenir le top 10 des gagnants pour une phase
function getTopWinnersForPhase($mysqli, $phase_name) {
    // Récupérer tous les utilisateurs
    $users = [];
    $result = $mysqli->query("SELECT id, username, ville, departement, photo FROM Users WHERE role = 'user' ORDER BY username");
    while($row = $result->fetch_assoc()) {
        $users[$row['id']] = $row;
    }
    $result->close();
    
    // Calculer les scores pour cette phase (SANS bonus)
    $scores = [];
    foreach ($users as $user_id => $user) {
        $score = getUserScoreForPhase($mysqli, $user_id, $phase_name);
        // Inclure uniquement les utilisateurs avec des points > 0
        if ($score > 0) {
            // Récupérer le temps passé dans le quiz pour cette phase
            $quiz_time = getQuizTimeForPhase($mysqli, $user_id, $phase_name);
            
            $scores[] = [
                'user_id' => $user_id,
                'username' => $user['username'],
                'ville' => $user['ville'],
                'departement' => $user['departement'],
                'photo' => $user['photo'],
                'score' => $score,
                'quiz_time' => $quiz_time
            ];
        }
    }
    
    // Trier par score décroissant, puis par temps passé dans le quiz croissant (plus rapide = mieux)
    usort($scores, function($a, $b) {
        if ($b['score'] === $a['score']) {
            // Si même score, trier par temps passé dans le quiz (le plus rapide en premier)
            if ($a['quiz_time'] === $b['quiz_time']) {
                return strcmp($a['username'], $b['username']);
            }
            return $a['quiz_time'] <=> $b['quiz_time'];
        }
        return $b['score'] <=> $a['score'];
    });
    
    // Retourner le top 3 (ou moins s'il y a moins de 3 utilisateurs avec des points)
    return array_slice($scores, 0, 3);
}

// Fonction pour obtenir le temps total passé dans tous les quiz
function getTotalQuizTime($mysqli, $user_id) {
    // Récupérer la somme de tous les temps de quiz pour cet utilisateur
    $stmt = $mysqli->prepare('SELECT SUM(TIMESTAMPDIFF(MICROSECOND, start_time, completed_at)) as total_time_diff FROM quizz_results_phases WHERE user_id = ? AND start_time IS NOT NULL AND completed_at IS NOT NULL');
    $stmt->bind_param('i', $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $stmt->close();
    
    if ($row && isset($row['total_time_diff']) && $row['total_time_diff'] !== null) {
        // Convertir les microsecondes en millisecondes (diviser par 1000)
        $milliseconds = intval($row['total_time_diff'] / 1000);
        return $milliseconds;
    }
    
    // Si pas de temps disponible, retourner une valeur très élevée pour être en dernier
    return PHP_INT_MAX;
}

// Fonction pour obtenir le top 3 des gagnants au total
function getTopWinnersTotal($mysqli) {
    // Récupérer tous les utilisateurs
    $users = [];
    $result = $mysqli->query("SELECT id, username, ville, departement, photo FROM Users WHERE role = 'user' ORDER BY username");
    while($row = $result->fetch_assoc()) {
        $users[$row['id']] = $row;
    }
    $result->close();
    
    // Calculer les scores totaux (toutes phases + bonus)
    $scores = [];
    foreach ($users as $user_id => $user) {
        $score = getUserTotalScore($mysqli, $user_id);
        // Inclure uniquement les utilisateurs avec des points > 0
        if ($score > 0) {
            // Récupérer le temps total passé dans tous les quiz
            $total_quiz_time = getTotalQuizTime($mysqli, $user_id);
            
            $scores[] = [
                'user_id' => $user_id,
                'username' => $user['username'],
                'ville' => $user['ville'],
                'departement' => $user['departement'],
                'photo' => $user['photo'],
                'score' => $score,
                'quiz_time' => $total_quiz_time
            ];
        }
    }
    
    // Trier par score décroissant, puis par temps total passé dans les quiz croissant (plus rapide = mieux)
    usort($scores, function($a, $b) {
        if ($b['score'] === $a['score']) {
            // Si même score, trier par temps total passé dans les quiz (le plus rapide en premier)
            if ($a['quiz_time'] === $b['quiz_time']) {
                return strcmp($a['username'], $b['username']);
            }
            return $a['quiz_time'] <=> $b['quiz_time'];
        }
        return $b['score'] <=> $a['score'];
    });
    
    // Retourner le top 3 (ou moins s'il y a moins de 3 utilisateurs avec des points)
    return array_slice($scores, 0, 3);
}

// Fonction pour récupérer les phases actives des gagnants
function getActiveGagnantsPhases($mysqli) {
    $phases = [];
    
    // Vérifier si la table existe
    $table_check = $mysqli->query("SHOW TABLES LIKE 'phase_management'");
    if ($table_check->num_rows == 0) {
        // Si la table n'existe pas, utiliser les phases par défaut
        return [
            [
                'phase' => 'Phase de groupes',
                'title' => 'Phase de groupes',
                'description' => 'Total des 3 journées de la phase de groupes'
            ],
            [
                'phase' => '8e de finale',
                'title' => '8e de finale',
                'description' => '8e de finale'
            ],
            [
                'phase' => 'Quart de finale',
                'title' => 'Quart de finale',
                'description' => 'Quart de finale'
            ],
            [
                'phase' => 'Demi-finale',
                'title' => 'Demi-finale',
                'description' => 'Demi-finale'
            ],
            [
                'phase' => 'Match pour la 3e place & Finale',
                'title' => 'Match pour la 3e place & Finale',
                'description' => 'Match pour la 3e place & Finale (cumulé)'
            ]
        ];
    }
    
    $stmt = $mysqli->prepare("SELECT phase_name FROM phase_management WHERE page_type = 'gagnants' AND is_active = 1 ORDER BY id");
    if (!$stmt) {
        // En cas d'erreur, utiliser les phases par défaut
        return [
            [
                'phase' => 'Phase de groupes',
                'title' => 'Phase de groupes',
                'description' => 'Total des 3 journées de la phase de groupes'
            ],
            [
                'phase' => '8e de finale',
                'title' => '8e de finale',
                'description' => '8e de finale'
            ],
            [
                'phase' => 'Quart de finale',
                'title' => 'Quart de finale',
                'description' => 'Quart de finale'
            ],
            [
                'phase' => 'Demi-finale',
                'title' => 'Demi-finale',
                'description' => 'Demi-finale'
            ],
            [
                'phase' => 'Match pour la 3e place & Finale',
                'title' => 'Match pour la 3e place & Finale',
                'description' => 'Match pour la 3e place & Finale (cumulé)'
            ]
        ];
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    $has_third_place = false;
    $has_finale = false;
    
    while ($row = $result->fetch_assoc()) {
        $phase_name = $row['phase_name'];
        if ($phase_name === 'Match pour la 3e place') {
            $has_third_place = true;
        } elseif ($phase_name === 'Finale') {
            $has_finale = true;
        } else {
            // Ajouter les autres phases normalement
            $phases[] = [
                'phase' => $phase_name,
                'title' => $phase_name,
                'description' => $phase_name === 'Phase de groupes' ? 'Total des 3 journées de la phase de groupes' : $phase_name
            ];
        }
    }
    
    $stmt->close();
    
    // Vérifier spécifiquement si "Finale" est active pour la page gagnants
    // L'onglet combiné ne sera affiché que si "Finale" est active
    $finale_is_active = false;
    $stmt_finale = $mysqli->prepare("SELECT is_active FROM phase_management WHERE page_type = 'gagnants' AND phase_name = 'Finale'");
    if ($stmt_finale) {
        $stmt_finale->execute();
        $result_finale = $stmt_finale->get_result();
        $row_finale = $result_finale->fetch_assoc();
        $stmt_finale->close();
        
        if ($row_finale && $row_finale['is_active'] == 1) {
            $finale_is_active = true;
        }
    }
    
    // Combiner Match pour la 3e place et Finale en un seul onglet
    // MAIS seulement si "Finale" est active
    if ($finale_is_active) {
        $phases[] = [
            'phase' => 'Match pour la 3e place & Finale',
            'title' => 'Match pour la 3e place & Finale',
            'description' => 'Match pour la 3e place & Finale (cumulé)'
        ];
    }
    
    // Si aucune phase active trouvée, utiliser les phases par défaut
    if (empty($phases)) {
        return [
            [
                'phase' => 'Phase de groupes',
                'title' => 'Phase de groupes',
                'description' => 'Total des 3 journées de la phase de groupes'
            ],
            [
                'phase' => '8e de finale',
                'title' => '8e de finale',
                'description' => '8e de finale'
            ],
            [
                'phase' => 'Quart de finale',
                'title' => 'Quart de finale',
                'description' => 'Quart de finale'
            ],
            [
                'phase' => 'Demi-finale',
                'title' => 'Demi-finale',
                'description' => 'Demi-finale'
            ],
            [
                'phase' => 'Match pour la 3e place & Finale',
                'title' => 'Match pour la 3e place & Finale',
                'description' => 'Match pour la 3e place & Finale (cumulé)'
            ]
        ];
    }
    
    return $phases;
}

// Fonction helper pour traduire le nom de la phase
function translatePhaseName($phase_name) {
    switch($phase_name) {
        case 'Phase de groupes':
            return __('group_phase');
        case '8e de finale':
            return __('round_of_16');
        case 'Quart de finale':
            return __('quarter_final');
        case 'Demi-finale':
            return __('semi_final');
        case 'Match pour la 3e place':
            return __('third_place_match');
        case 'Finale':
            return __('final');
        case 'Match pour la 3e place & Finale':
            return __('third_place_match') . ' & ' . __('final');
        case 'Total':
            return __('general_total');
        default:
            return $phase_name;
    }
}

// Récupérer les phases actives
$phases_sorted = getActiveGagnantsPhases($mysqli);

// Ajouter l'onglet "Total" à la fin
$phases_sorted[] = [
    'phase' => 'Total',
    'title' => 'Total',
    'description' => 'Classement général (toutes phases + groupes + vainqueur CAN + quiz)'
];

$current_lang = getCurrentLanguage();
?>
<style>
.winners-container {
    max-width: 900px;
    margin: 0 auto;
    padding: 0 12px;
}

.phase-section {
    background: #23232b;
    border-radius: 18px;
    box-shadow: 0 4px 24px #0005;
    padding: 32px 24px 24px 24px;
    margin-bottom: 30px;
}

.phase-title {
    color: #ffb300;
    font-size: 1.8rem;
    font-weight: 800;
    text-align: center;
    margin-bottom: 10px;
    border-bottom: 2px solid #ffb300;
    padding-bottom: 10px;
}

.phase-description {
    color: #ffffff;
    font-size: 1.1rem;
    text-align: center;
    margin-bottom: 25px;
    opacity: 0.9;
    font-style: italic;
}

/* Styles pour les tabs des gagnants */
.winners-tabs {
    border-bottom: 2px solid rgba(255, 179, 0, 0.3);
    flex-wrap: nowrap;
    white-space: nowrap;
}

.winners-tabs .nav-item {
    flex-shrink: 0;
}

.winners-tabs .nav-link {
    color: #ffffff;
    background: transparent;
    border: none;
    border-radius: 8px 8px 0 0;
    padding: 12px 16px;
    font-weight: 600;
    transition: all 0.3s ease;
    white-space: nowrap;
    font-size: 0.9rem;
}

.winners-tabs .nav-link:hover {
    color: #ffb300;
    background: rgba(255, 179, 0, 0.1);
}

.winners-tabs .nav-link.active {
    color: #ffb300;
    background: rgba(255, 179, 0, 0.2);
    border-bottom: 2px solid #ffb300;
}

/* Responsive pour les onglets */
@media (max-width: 768px) {
    .winners-tabs .nav-link {
        padding: 10px 12px;
        font-size: 0.8rem;
    }
    
    .winners-container {
        padding: 0 8px;
        margin-top: 50px;
    }
    
    .phase-section {
        padding: 20px 15px;
        margin-bottom: 20px;
    }
    
    .phase-title {
        font-size: 1.4rem;
        margin-bottom: 8px;
    }
    
    .phase-description {
        font-size: 1rem;
        margin-bottom: 20px;
    }
    
    .winner-card {
        padding: 15px;
        margin-bottom: 12px;
        flex-direction: column;
        text-align: center;
        gap: 15px;
    }
    
    .winner-rank {
        font-size: 1.5rem;
        min-width: auto;
    }
    
    .winner-photo {
        width: 50px;
        height: 50px;
    }
    
    .winner-name {
        font-size: 1.1rem;
    }
    
    .winner-location {
        font-size: 0.9rem;
    }
    
    .winner-score {
        font-size: 1.3rem;
        min-width: auto;
    }
    
    .winner-details {
        flex-direction: column;
        gap: 8px;
        margin-top: 5px;
    }
    
    .winner-detail-item {
        font-size: 0.85rem;
    }
    
    .tab-content {
        padding: 15px;
        min-height: 300px;
    }
    
    .winners-tabs {
        overflow-x: auto;
        white-space: nowrap;
        scrollbar-width: thin;
        scrollbar-color: #ffb300 #2a2a3a;
    }
    
    .winners-tabs::-webkit-scrollbar {
        height: 6px;
    }
    
    .winners-tabs::-webkit-scrollbar-track {
        background: #2a2a3a;
        border-radius: 3px;
    }
    
    .winners-tabs::-webkit-scrollbar-thumb {
        background: #ffb300;
        border-radius: 3px;
    }
    
    .winners-tabs::-webkit-scrollbar-thumb:hover {
        background: #ffc107;
    }
}

.tab-content {
    border-radius: 0 0 15px 15px;
    padding: 0;
    min-height: 400px;
    max-height: 100vh;
    overflow-y: visible;
    scrollbar-width: thin;
    scrollbar-color: #ffb300 #2a2a3a;
}

/* Styles pour la scrollbar WebKit (Chrome, Safari, Edge) */
.tab-content::-webkit-scrollbar {
    width: 8px;
}

.tab-content::-webkit-scrollbar-track {
    background: #2a2a3a;
    border-radius: 4px;
}

.tab-content::-webkit-scrollbar-thumb {
    background: #ffb300;
    border-radius: 4px;
}

.tab-content::-webkit-scrollbar-thumb:hover {
    background: #ffc107;
}

.winner-card {
    background: #2a2a3a;
    border-radius: 12px;
    padding: 20px;
    margin-bottom: 15px;
    border-left: 4px solid #ffb300;
    display: flex;
    align-items: center;
    gap: 20px;
    transition: transform 0.2s, box-shadow 0.2s;
}

.winner-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(255, 179, 0, 0.2);
}

.winner-card.first {
    border-left-color: #ffd700;
    background: linear-gradient(135deg, #2a2a3a 0%, #3a3a4a 100%);
}

.winner-card.second {
    border-left-color: #c0c0c0;
}

.winner-card.third {
    border-left-color: #cd7f32;
}

.winner-rank {
    font-size: 2rem;
    font-weight: bold;
    color: #ffb300;
    min-width: 60px;
    text-align: center;
}

.winner-rank.first {
    color: #ffd700;
}

.winner-rank.second {
    color: #c0c0c0;
}

.winner-rank.third {
    color: #cd7f32;
}

.winner-photo {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    object-fit: cover;
    border: 3px solid #ffb300;
}

.winner-info {
    flex: 1;
}

.winner-name {
    color: #fff;
    font-size: 1.3rem;
    font-weight: 700;
    margin-bottom: 5px;
}

.winner-location {
    color: #ccc;
    font-size: 1rem;
    margin-bottom: 5px;
}

.winner-score {
    color: #ffb300;
    font-size: 1.5rem;
    font-weight: bold;
}

.no-winners {
    text-align: center;
    color: #888;
    font-style: italic;
    padding: 40px;
}

/* Styles RTL pour l'arabe */
.lang-ar .winner-card {
    border-left: none;
    border-right: 4px solid #ffb300;
}

.lang-ar .winner-card.first {
    border-right-color: #ffd700;
}

.lang-ar .winner-card.second {
    border-right-color: #c0c0c0;
}

.lang-ar .winner-card.third {
    border-right-color: #cd7f32;
}

.lang-ar .winner-info {
    text-align: right;
}

.lang-ar .phase-title {
    text-align: center;
}

/* Styles pour les détails des gagnants */
.winner-details {
    display: flex;
    flex-direction: row;
    gap: 15px;
    margin-top: 8px;
    flex-wrap: wrap;
}

.winner-detail-item {
    display: flex;
    align-items: center;
    gap: 5px;
    color: #cccccc;
    font-size: 0.9rem;
}

.detail-label {
    font-weight: 600;
    color: #ffb300;
}

.detail-value {
    color: #ffffff;
    font-weight: 500;
}

.winner-score {
    font-size: 1.5rem;
    font-weight: 800;
    color: #ffb300;
    text-align: center;
    min-width: 80px;
}

/* Styles RTL pour les détails */
.lang-ar .winner-detail-item {
    flex-direction: row-reverse;
    text-align: right;
}

.lang-ar .detail-label {
    min-width: 100px;
}
</style>

<div class="winners-container page-gagnants" style="margin-top: 30px;">
    <h1 class="text-center mb-4" style="color: #ffffff; font-weight: 800;">🏆 <?= __('winners_by_phase') ?></h1>
    
    <!-- Tabs pour les phases -->
    <ul class="nav nav-tabs winners-tabs mb-4" id="winnersTabs" role="tablist">
        <?php foreach ($phases_sorted as $index => $phase): ?>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?= $index === 0 ? 'active' : '' ?>" 
                        id="tab-<?= $index ?>" 
                        data-bs-toggle="tab" 
                        data-bs-target="#phase-<?= $index ?>" 
                        type="button" 
                        role="tab">
                    <?= htmlspecialchars(translatePhaseName($phase['phase'])) ?>
                </button>
            </li>
        <?php endforeach; ?>
    </ul>
    
    <!-- Contenu des tabs -->
    <div class="tab-content" id="winnersTabContent">
        <?php foreach ($phases_sorted as $index => $phase): ?>
            <?php 
            // Utiliser la fonction appropriée selon le type d'onglet
            if ($phase['phase'] === 'Total') {
                $winners = getTopWinnersTotal($mysqli);
            } else {
                $winners = getTopWinnersForPhase($mysqli, $phase['phase']);
            }
            ?>
            <div class="tab-pane fade <?= $index === 0 ? 'show active' : '' ?>" 
                 id="phase-<?= $index ?>" 
                 role="tabpanel">
                
                <div class="phase-section">
                    <h2 class="phase-title"><?= __('top_3_general') ?> : <?= htmlspecialchars(translatePhaseName($phase['phase'])) ?></h2>
                    
                    <?php if (empty($winners)): ?>
                        <div class="no-winners">
                            <i class="fas fa-trophy" style="font-size: 3rem; color: #666; margin-bottom: 15px;"></i>
                            <p><?= __('no_winners_phase') ?></p>
                            <p class="text-muted"><?= __('scores_will_display') ?></p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($winners as $winner_index => $winner): ?>
                            <?php 
                            $rank_class = '';
                            $rank_icon = '';
                            $rank_number = $winner_index + 1;
                            if ($winner_index === 0) {
                                $rank_class = 'first';
                                $rank_icon = '🥇';
                            } elseif ($winner_index === 1) {
                                $rank_class = 'second';
                                $rank_icon = '🥈';
                            } elseif ($winner_index === 2) {
                                $rank_class = 'third';
                                $rank_icon = '🥉';
                            }
                            ?>
                            <div class="winner-card <?= $rank_class ?>">
                                <div class="winner-rank <?= $rank_class ?>">
                                    <?php if ($rank_icon): ?>
                                        <?= $rank_icon ?>
                                    <?php else: ?>
                                        <span style="font-size: 1.5rem; font-weight: bold; color: #ffb300;"><?= $rank_number ?></span>
                                    <?php endif; ?>
                                </div>
                                
                                <?php
                                // Générer les initiales si pas de photo
                                $initials = '';
                                if (empty($winner['photo'])) {
                                    $username = trim($winner['username']);
                                    $words = explode(' ', $username);
                                    if (count($words) >= 2) {
                                        // Prendre la première lettre du prénom et du nom
                                        $initials = strtoupper(substr($words[0], 0, 1) . substr($words[count($words) - 1], 0, 1));
                                    } elseif (count($words) == 1 && strlen($words[0]) > 0) {
                                        // Si un seul mot, prendre les deux premières lettres
                                        $initials = strtoupper(substr($words[0], 0, min(2, strlen($words[0]))));
                                    } else {
                                        $initials = 'AN';
                                    }
                                }
                                ?>
                                <?php if (!empty($winner['photo'])): ?>
                                    <img src="assets/img/upload/<?= htmlspecialchars($winner['photo']) ?>" 
                                         alt="Photo de <?= htmlspecialchars($winner['username']) ?>" 
                                         class="winner-photo">
                                <?php else: ?>
                                    <div class="winner-photo" style="display: flex; align-items: center; justify-content: center; background: linear-gradient(135deg, #667eea00 0%, #764ba200 100%); color: white; font-size: 1.5rem; font-weight: bold;">
                                        <?= htmlspecialchars($initials) ?>
                                    </div>
                                <?php endif; ?>
                                
                                <div class="winner-info">
                                    <div class="winner-name"><?= htmlspecialchars($winner['username']) ?></div>
                                    <div class="winner-details">
                                        <?php if ($winner['ville']): ?>
                                            <span class="winner-detail-item">
                                                <i class="fas fa-map-marker-alt"></i>
                                                <span class="detail-label"><?= __('city') ?> :</span>
                                                <span class="detail-value"><?= htmlspecialchars($winner['ville']) ?></span>
                                            </span>
                                        <?php endif; ?>
                                        <?php if ($winner['departement']): ?>
                                            <span class="winner-detail-item">
                                                <i class="fas fa-building"></i>
                                                <span class="detail-label"><?= __('department') ?> :</span>
                                                <span class="detail-value"><?= htmlspecialchars($winner['departement']) ?></span>
                                            </span>
                                        <?php endif; ?>
                                        <?php if (isset($winner['quiz_time'])): ?>
                                            <span class="winner-detail-item">
                                                <i class="fas fa-clock"></i>
                                                <span class="detail-label"><?= ($current_lang === 'ar') ? 'الوقت في الاختبار' : 'Temps quiz' ?> :</span>
                                                <span class="detail-value"><?= formatQuizTime($winner['quiz_time']) ?></span>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <div class="winner-score">
                                    <?= $winner['score'] ?> <?= __('points_label') ?>
                                </div>
                                
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<?php include 'templates/footer.php'; ?>
