<?php
date_default_timezone_set('UTC');

// Inclure le système de langue en premier
require_once 'includes/language.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

// Inclure la base de données
require_once 'includes/db.php';

// Définir les variables nécessaires
$current_lang = getCurrentLanguage();
$user_id = $_SESSION['user_id'];

// Fonction pour récupérer TOUTES les phases du quiz (actives et inactives)
function getAllQuizPhases($mysqli, $current_lang) {
    $phases = [];
    
    // Vérifier si la table existe
    $table_check = $mysqli->query("SHOW TABLES LIKE 'phase_management'");
    if ($table_check->num_rows == 0) {
        // Si la table n'existe pas, utiliser les phases par défaut
        // IMPORTANT: Ne pas inclure "Match pour la 3e place" car elle est combinée avec "Finale"
        return [
            'Phase de groupes' => ($current_lang === 'ar') ? 'مرحلة المجموعات' : 'Phase de groupes',
            '8e de finale' => ($current_lang === 'ar') ? 'دور الثمن النهائي' : '8e de finale',
            'Quart de finale' => ($current_lang === 'ar') ? 'ربع النهائي' : 'Quart de finale',
            'Demi-finale' => ($current_lang === 'ar') ? 'نصف النهائي' : 'Demi-finale',
            'Finale' => ($current_lang === 'ar') ? 'النهائي' : 'Finale'
        ];
    }
    
    // Récupérer TOUTES les phases (actives et inactives) depuis la base de données
    // Seules les phases qui existent dans la base seront affichées
    $stmt = $mysqli->prepare("SELECT phase_name FROM phase_management WHERE page_type = 'quiz' ORDER BY id");
    if (!$stmt) {
        // En cas d'erreur, retourner un tableau vide (pas de phases par défaut)
        return [];
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    // Liste de toutes les phases possibles avec leurs traductions
    // IMPORTANT: "Match pour la 3e place" n'est plus affichée séparément, elle est combinée avec "Finale"
    $all_possible_phases = [
        'Phase de groupes' => ($current_lang === 'ar') ? 'مرحلة المجموعات' : 'Phase de groupes',
        '8e de finale' => ($current_lang === 'ar') ? 'دور الثمن النهائي' : '8e de finale',
        'Quart de finale' => ($current_lang === 'ar') ? 'ربع النهائي' : 'Quart de finale',
        'Demi-finale' => ($current_lang === 'ar') ? 'نصف النهائي' : 'Demi-finale',
        'Match pour la 3e place' => ($current_lang === 'ar') ? 'مباراة المركز الثالث' : 'Match pour la 3e place',
        'Finale' => ($current_lang === 'ar') ? 'النهائي' : 'Finale'
    ];
    
    // Récupérer les phases depuis la base de données
    $phases_from_db = [];
    $has_third_place = false;
    $has_finale = false;
    
    while ($row = $result->fetch_assoc()) {
        $phase_key = $row['phase_name'];
        
        // Si c'est "Match pour la 3e place", on ne l'ajoute pas directement mais on note qu'elle existe
        if ($phase_key === 'Match pour la 3e place') {
            $has_third_place = true;
            continue; // Ne pas l'ajouter à la liste des phases affichées
        }
        
        // Si c'est "Finale", on note qu'elle existe
        if ($phase_key === 'Finale') {
            $has_finale = true;
        }
        
        if (isset($all_possible_phases[$phase_key])) {
            $phases_from_db[$phase_key] = $all_possible_phases[$phase_key];
        } else {
            // Si la phase n'est pas dans la liste, utiliser le nom tel quel
            $phases_from_db[$phase_key] = $phase_key;
        }
    }
    
    $stmt->close();
    
    // Si "Match pour la 3e place" existe dans la base mais pas "Finale", ajouter "Finale" quand même
    // pour permettre le quiz combiné
    if ($has_third_place && !$has_finale && !isset($phases_from_db['Finale'])) {
        $phases_from_db['Finale'] = $all_possible_phases['Finale'];
    }
    
    // Si "Finale" existe, elle sera toujours affichée (même si "Match pour la 3e place" n'existe pas)
    // car le quiz "Finale" combine les deux phases
    
    // Trier les phases dans l'ordre chronologique
    $phase_order = [
        'Phase de groupes' => 1,
        '8e de finale' => 2,
        'Quart de finale' => 3,
        'Demi-finale' => 4,
        'Finale' => 5
    ];
    
    uksort($phases_from_db, function($a, $b) use ($phase_order) {
        $order_a = $phase_order[$a] ?? 999;
        $order_b = $phase_order[$b] ?? 999;
        return $order_a - $order_b;
    });
    
    return $phases_from_db;
}

// Fonction pour vérifier si une phase est active
function isPhaseActive($mysqli, $phase_key) {
    // Pour "Finale", vérifier si l'une des deux phases (Match pour la 3e place ou Finale) est active
    if ($phase_key === 'Finale') {
        $table_check = $mysqli->query("SHOW TABLES LIKE 'phase_management'");
        if ($table_check->num_rows == 0) {
            return true; // Par défaut, toutes les phases sont actives si la table n'existe pas
        }
        
        // Vérifier si "Finale" est active
        $stmt = $mysqli->prepare("SELECT is_active FROM phase_management WHERE page_type = 'quiz' AND phase_name = 'Finale'");
        if ($stmt) {
            $stmt->execute();
            $result = $stmt->get_result();
            $row_finale = $result->fetch_assoc();
            $stmt->close();
            
            if ($row_finale && $row_finale['is_active'] == 1) {
                return true;
            }
        }
        
        // Vérifier si "Match pour la 3e place" est active (car elle est combinée avec Finale)
        $stmt = $mysqli->prepare("SELECT is_active FROM phase_management WHERE page_type = 'quiz' AND phase_name = 'Match pour la 3e place'");
        if ($stmt) {
            $stmt->execute();
            $result = $stmt->get_result();
            $row_third = $result->fetch_assoc();
            $stmt->close();
            
            if ($row_third && $row_third['is_active'] == 1) {
                return true;
            }
        }
        
        // Si aucune des deux n'est active ou n'existe pas, retourner false
        return false;
    }
    
    // Pour les autres phases, logique normale
    $table_check = $mysqli->query("SHOW TABLES LIKE 'phase_management'");
    if ($table_check->num_rows == 0) {
        return true; // Par défaut, toutes les phases sont actives si la table n'existe pas
    }
    
    $stmt = $mysqli->prepare("SELECT is_active FROM phase_management WHERE page_type = 'quiz' AND phase_name = ?");
    if (!$stmt) {
        return false; // Si erreur de préparation, considérer comme inactif
    }
    
    $stmt->bind_param('s', $phase_key);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $stmt->close();
    
    // Si la phase n'existe pas dans la base, retourner false (inactive)
    if (!$row) {
        return false;
    }
    
    return ($row['is_active'] == 1);
}

// Récupérer toutes les phases (actives et inactives)
$phases = getAllQuizPhases($mysqli, $current_lang);

// Fonction pour récupérer une question traduite
function getTranslatedQuestion($mysqli, $question_id, $current_lang) {
    $table_name = ($current_lang === 'ar') ? 'quizz_questions_ar' : 'quizz_questions';
    $stmt = $mysqli->prepare("SELECT * FROM {$table_name} WHERE id = ?");
    $stmt->bind_param('i', $question_id);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    return $result;
}

// Fonction pour récupérer 15 questions aléatoires pour une phase (sans filtrage phase/thème)
// Exclut les questions déjà posées à l'utilisateur dans d'autres phases
function getQuestionsForPhase($mysqli, $user_id, $phase, $current_lang, $use_session = true) {
    // Vérifier si les questions sont déjà en session pour cette phase
    if ($use_session && isset($_SESSION['quiz_questions'][$phase])) {
        return $_SESSION['quiz_questions'][$phase];
    }
    
    $table_name = ($current_lang === 'ar') ? 'quizz_questions_ar' : 'quizz_questions';
    
    // Vérifier si les colonnes existent, sinon les créer
    $check_cols = $mysqli->query("SHOW COLUMNS FROM quizz_results_phases LIKE 'current_question_id'");
    if ($check_cols->num_rows == 0) {
        $mysqli->query("ALTER TABLE quizz_results_phases ADD COLUMN current_question_id INT NULL, ADD COLUMN current_question_idx INT NULL");
    }
    
    // Vérifier d'abord combien de questions sont disponibles dans la base
    $total_count_result = $mysqli->query("SELECT COUNT(*) as total FROM {$table_name}");
    $total_count = 0;
    if ($total_count_result) {
        $row = $total_count_result->fetch_assoc();
        $total_count = intval($row['total']);
    }
    
    // Si moins de 15 questions dans la base, retourner toutes les questions disponibles
    if ($total_count < 15) {
        $sql = "SELECT * FROM {$table_name} ORDER BY RAND()";
        $result = $mysqli->query($sql);
        $questions = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $questions[] = $row;
            }
        }
        
        // Stocker en session
        if ($use_session) {
            if (!isset($_SESSION['quiz_questions'])) {
                $_SESSION['quiz_questions'] = [];
            }
            $_SESSION['quiz_questions'][$phase] = $questions;
        }
        
        return $questions;
    }
    
    // Récupérer TOUTES les questions déjà répondues par cet utilisateur (toutes phases confondues)
    // Utiliser la table quiz_answered_questions si elle existe, sinon fallback sur current_question_id
    $used_question_ids = [];
    
    // Vérifier si la table quiz_answered_questions existe
    $table_check = $mysqli->query("SHOW TABLES LIKE 'quiz_answered_questions'");
    if ($table_check && $table_check->num_rows > 0) {
        // Méthode optimale : utiliser la table de suivi
        // IMPORTANT: Exclure TOUTES les questions déjà répondues par cet utilisateur, peu importe la phase
        $stmt = $mysqli->prepare("
            SELECT DISTINCT question_id 
            FROM quiz_answered_questions 
            WHERE user_id = ?
        ");
        if ($stmt) {
            $stmt->bind_param('i', $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                if ($row['question_id']) {
                    $used_question_ids[] = intval($row['question_id']);
                }
            }
            $stmt->close();
        }
    } else {
        // Fallback : utiliser current_question_id (méthode limitée)
        // Récupérer toutes les questions de toutes les phases pour cet utilisateur
        $check_cols = $mysqli->query("SHOW COLUMNS FROM quizz_results_phases LIKE 'current_question_id'");
        if ($check_cols && $check_cols->num_rows > 0) {
            $stmt = $mysqli->prepare("
                SELECT DISTINCT qr.current_question_id 
                FROM quizz_results_phases qr 
                WHERE qr.user_id = ? AND qr.current_question_id IS NOT NULL
            ");
            if ($stmt) {
                $stmt->bind_param('i', $user_id);
                $stmt->execute();
                $result = $stmt->get_result();
                while ($row = $result->fetch_assoc()) {
                    if ($row['current_question_id']) {
                        $used_question_ids[] = intval($row['current_question_id']);
                    }
                }
                $stmt->close();
            }
        }
    }
    
    // Log pour debug (peut être supprimé en production)
    // error_log("User {$user_id} - Phase {$phase}: " . count($used_question_ids) . " questions déjà posées exclues");
    
    // Construire la requête pour exclure les questions déjà utilisées
    $questions = [];
    
    // Nettoyer et valider les IDs des questions déjà posées
    $used_question_ids = array_unique(array_map('intval', $used_question_ids));
    $used_question_ids = array_filter($used_question_ids, function($id) {
        return $id > 0;
    });
    
    // TOUJOURS exclure les questions déjà posées
    if (!empty($used_question_ids)) {
        // Construire la liste des IDs pour la requête SQL (sécurisée car tous les IDs sont des entiers)
        $ids_str = implode(',', $used_question_ids);
        
        // Récupérer les questions qui ne sont PAS dans la liste des questions déjà posées
        $sql = "SELECT * FROM {$table_name} WHERE id NOT IN ({$ids_str}) ORDER BY RAND() LIMIT 15";
        $result = $mysqli->query($sql);
        
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                // Vérification supplémentaire : s'assurer que cette question n'est pas dans la liste
                if (!in_array(intval($row['id']), $used_question_ids)) {
                    $questions[] = $row;
                }
            }
        }
    } else {
        // Aucune question utilisée, récupérer 15 questions aléatoires
        $sql = "SELECT * FROM {$table_name} ORDER BY RAND() LIMIT 15";
        $result = $mysqli->query($sql);
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $questions[] = $row;
            }
        }
    }
    
    // Vérification finale de sécurité : s'assurer qu'aucune question déjà posée n'est dans la liste
    if (!empty($used_question_ids) && !empty($questions)) {
        $question_ids = array_column($questions, 'id');
        $duplicates = array_intersect($question_ids, $used_question_ids);
        if (!empty($duplicates)) {
            // Supprimer les doublons trouvés (ne devrait jamais arriver, mais sécurité supplémentaire)
            $questions = array_filter($questions, function($q) use ($used_question_ids) {
                return !in_array(intval($q['id']), $used_question_ids);
            });
            $questions = array_values($questions); // Réindexer le tableau
        }
    }
    
    // Limiter à 15 questions exactement (ou moins si pas assez disponibles)
    $questions = array_slice($questions, 0, 15);
    
    // Stocker les questions en session pour éviter les répétitions
    if ($use_session) {
        if (!isset($_SESSION['quiz_questions'])) {
            $_SESSION['quiz_questions'] = [];
        }
        $_SESSION['quiz_questions'][$phase] = $questions;
    }
    
    return $questions;
}

// Fonction pour vérifier l'état d'un quiz pour une phase
function getQuizState($mysqli, $user_id, $phase) {
    // Pour "Finale", vérifier d'abord "Finale", puis "Match pour la 3e place" (car combinées)
    if ($phase === 'Finale') {
        // D'abord vérifier "Finale"
        $phase_finale = 'Finale';
        $stmt = $mysqli->prepare('SELECT score, locked, start_time, completed_at FROM quizz_results_phases WHERE user_id = ? AND phase = ?');
        $stmt->bind_param('is', $user_id, $phase_finale);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        // Si "Finale" existe, retourner ce résultat (même si non complété)
        // Cela permet de continuer un quiz "Finale" qui vient d'être démarré
        if ($result) {
            return $result;
        }
        
        // Sinon, vérifier "Match pour la 3e place"
        $phase_third = 'Match pour la 3e place';
        $stmt = $mysqli->prepare('SELECT score, locked, start_time, completed_at FROM quizz_results_phases WHERE user_id = ? AND phase = ?');
        $stmt->bind_param('is', $user_id, $phase_third);
        $stmt->execute();
        $result_third = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        // Si "Match pour la 3e place" existe, utiliser ce résultat (même si non complété)
        // Cela permet de continuer un quiz commencé pour "Match pour la 3e place" dans "Finale"
        if ($result_third) {
            return $result_third;
        }
        
        // Si aucun des deux n'existe, retourner état vide
        return ['score' => 0, 'locked' => 0, 'start_time' => null, 'completed_at' => null];
    }
    
    // Pour les autres phases, logique normale
    $stmt = $mysqli->prepare('SELECT score, locked, start_time, completed_at FROM quizz_results_phases WHERE user_id = ? AND phase = ?');
    $stmt->bind_param('is', $user_id, $phase);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    return $result ?: ['score' => 0, 'locked' => 0, 'start_time' => null, 'completed_at' => null];
}

// Fonction pour récupérer le temps restant depuis la base de données
function getTimerFromDB($mysqli, $user_id, $phase) {
    $stmt = $mysqli->prepare('SELECT start_time, time_left FROM quiz_timer_sessions WHERE user_id = ? AND phase = ?');
    $stmt->bind_param('is', $user_id, $phase);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $stmt->close();
    
    if ($row) {
        $current_time = time();
        $elapsed = $current_time - $row['start_time'];
        $time_left = max(0, $row['time_left'] - $elapsed);
        
        // Mettre à jour le temps restant dans la base
        $stmt = $mysqli->prepare('UPDATE quiz_timer_sessions SET time_left = ? WHERE user_id = ? AND phase = ?');
        $stmt->bind_param('iis', $time_left, $user_id, $phase);
        $stmt->execute();
        $stmt->close();
        
        return $time_left;
    }
    
    return 300; // 5 minutes par défaut
}

// Gestion de la sélection de phase
if (isset($_POST['select_phase'])) {
    $selected_phase = $_POST['phase'];
    $_SESSION['selected_phase'] = $selected_phase;
    header('Location: quizz_new_phases.php');
    exit;
}

// Gestion du démarrage d'un quiz
if (isset($_POST['start_quiz'])) {
    $phase = $_POST['phase'];
    
    // Si on démarre le quiz "Finale", migrer les données de "Match pour la 3e place" si elles existent
    if ($phase === 'Finale') {
        // Vérifier s'il y a un quiz existant pour "Match pour la 3e place"
        $phase_third = 'Match pour la 3e place';
        $stmt_check = $mysqli->prepare('SELECT score, locked, start_time, completed_at, current_question_id, current_question_idx FROM quizz_results_phases WHERE user_id = ? AND phase = ?');
        $stmt_check->bind_param('is', $user_id, $phase_third);
        $stmt_check->execute();
        $result_third = $stmt_check->get_result()->fetch_assoc();
        $stmt_check->close();
        
        if ($result_third) {
            // Migrer les données vers "Finale"
            $phase_finale = 'Finale';
            // Si le quiz "Match pour la 3e place" est complété, migrer le score
            if ($result_third['locked'] == 1) {
                $stmt_migrate = $mysqli->prepare('INSERT INTO quizz_results_phases (user_id, phase, score, locked, completed_at, start_time) VALUES (?, ?, ?, 1, ?, ?) ON DUPLICATE KEY UPDATE score = VALUES(score), locked = 1, completed_at = VALUES(completed_at)');
                $stmt_migrate->bind_param('isiss', $user_id, $phase_finale, $result_third['score'], $result_third['completed_at'], $result_third['start_time']);
                $stmt_migrate->execute();
                $stmt_migrate->close();
            } else {
                // Si le quiz n'est pas complété, migrer l'état en cours
                $stmt_migrate = $mysqli->prepare('INSERT INTO quizz_results_phases (user_id, phase, start_time, current_question_id, current_question_idx, locked) VALUES (?, ?, ?, ?, ?, 0) ON DUPLICATE KEY UPDATE start_time = VALUES(start_time), current_question_id = VALUES(current_question_id), current_question_idx = VALUES(current_question_idx), locked = 0');
                $stmt_migrate->bind_param('isiii', $user_id, $phase_finale, $result_third['start_time'], $result_third['current_question_id'], $result_third['current_question_idx']);
                $stmt_migrate->execute();
                $stmt_migrate->close();
            }
            
            // Migrer aussi le timer si nécessaire
            $stmt_timer = $mysqli->prepare('SELECT start_time, time_left FROM quiz_timer_sessions WHERE user_id = ? AND phase = ?');
            $stmt_timer->bind_param('is', $user_id, $phase_third);
            $stmt_timer->execute();
            $timer_result = $stmt_timer->get_result()->fetch_assoc();
            $stmt_timer->close();
            
            if ($timer_result) {
                $stmt_timer_migrate = $mysqli->prepare('INSERT INTO quiz_timer_sessions (user_id, phase, start_time, time_left) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE start_time = VALUES(start_time), time_left = VALUES(time_left)');
                $stmt_timer_migrate->bind_param('isii', $user_id, $phase_finale, $timer_result['start_time'], $timer_result['time_left']);
                $stmt_timer_migrate->execute();
                $stmt_timer_migrate->close();
            }
        }
    }
    
    $quiz_state = getQuizState($mysqli, $user_id, $phase);
    
    // Vérifier si le quiz n'est pas déjà terminé
    if ($quiz_state['locked'] == 1) {
        $_SESSION['error'] = ($current_lang === 'ar') ? 'تم إكمال هذا الاختبار بالفعل' : 'Ce quiz a déjà été complété';
        header('Location: quizz_new_phases.php');
        exit;
    }
    
    // Vérifier AVANT de démarrer qu'il y a au moins 15 questions disponibles non déjà répondues
    $table_name = ($current_lang === 'ar') ? 'quizz_questions_ar' : 'quizz_questions';
    
    // Récupérer toutes les questions déjà répondues par cet utilisateur
    $used_question_ids = [];
    $table_check = $mysqli->query("SHOW TABLES LIKE 'quiz_answered_questions'");
    if ($table_check && $table_check->num_rows > 0) {
        $stmt = $mysqli->prepare("
            SELECT DISTINCT question_id 
            FROM quiz_answered_questions 
            WHERE user_id = ?
        ");
        if ($stmt) {
            $stmt->bind_param('i', $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                if ($row['question_id']) {
                    $used_question_ids[] = intval($row['question_id']);
                }
            }
            $stmt->close();
        }
    }
    
    // Compter les questions disponibles (non déjà répondues)
    $available_count = 0;
    if (!empty($used_question_ids)) {
        $ids_str = implode(',', array_map('intval', $used_question_ids));
        $count_result = $mysqli->query("SELECT COUNT(*) as total FROM {$table_name} WHERE id NOT IN ({$ids_str})");
        if ($count_result) {
            $row = $count_result->fetch_assoc();
            $available_count = intval($row['total']);
        }
    } else {
        // Aucune question utilisée, toutes les questions sont disponibles
        $count_result = $mysqli->query("SELECT COUNT(*) as total FROM {$table_name}");
        if ($count_result) {
            $row = $count_result->fetch_assoc();
            $available_count = intval($row['total']);
        }
    }
    
    // Vérifier qu'il y a au moins 15 questions disponibles
    if ($available_count < 15) {
        $_SESSION['error'] = ($current_lang === 'ar') 
            ? "لا توجد أسئلة كافية متاحة. لديك {$available_count} أسئلة متاحة فقط. تحتاج إلى 15 سؤال على الأقل." 
            : "Pas assez de questions disponibles. Vous avez seulement {$available_count} questions disponibles. Il faut au moins 15 questions.";
        header('Location: quizz_new_phases.php');
        exit;
    }
    
    // Récupérer les 15 questions aléatoires pour cette phase (elles seront toutes nouvelles)
    $questions = getQuestionsForPhase($mysqli, $user_id, $phase, $current_lang, true);
    
    if (empty($questions) || count($questions) < 15) {
        $_SESSION['error'] = ($current_lang === 'ar') 
            ? 'خطأ: لم يتم العثور على 15 سؤال متاح' 
            : 'Erreur: Impossible de trouver 15 questions disponibles';
        header('Location: quizz_new_phases.php');
        exit;
    }
    
    // Démarrer le quiz
    // Vérifier que la première question existe
    if (!isset($questions[0]) || !isset($questions[0]['id'])) {
        $_SESSION['error'] = ($current_lang === 'ar') ? 'خطأ في تحميل الأسئلة' : 'Erreur lors du chargement des questions';
        header('Location: quizz_new_phases.php');
        exit;
    }
    
    // Vérifier que toutes les questions sont bien nouvelles (non déjà répondues)
    // Double vérification de sécurité
    $question_ids = array_column($questions, 'id');
    if (!empty($used_question_ids)) {
        $duplicates = array_intersect($question_ids, $used_question_ids);
        if (!empty($duplicates)) {
            $_SESSION['error'] = ($current_lang === 'ar') 
                ? 'خطأ: تم العثور على أسئلة مكررة. يرجى المحاولة مرة أخرى.' 
                : 'Erreur: Des questions dupliquées ont été trouvées. Veuillez réessayer.';
            header('Location: quizz_new_phases.php');
            exit;
        }
    }
    
    // Enregistrer toutes les 15 questions sélectionnées dans quiz_answered_questions
    // AVANT de démarrer le quiz pour éviter qu'elles soient proposées dans d'autres phases
    $table_check = $mysqli->query("SHOW TABLES LIKE 'quiz_answered_questions'");
    if ($table_check && $table_check->num_rows > 0) {
        $enregistrees = 0;
        foreach ($questions as $q) {
            $track_stmt = $mysqli->prepare("
                INSERT IGNORE INTO quiz_answered_questions (user_id, question_id, phase)
                VALUES (?, ?, ?)
            ");
            if ($track_stmt) {
                $track_stmt->bind_param('iis', $user_id, $q['id'], $phase);
                if ($track_stmt->execute()) {
                    $enregistrees++;
                }
                $track_stmt->close();
            }
        }
        
        // Vérifier que toutes les questions ont été enregistrées
        if ($enregistrees < count($questions)) {
            $_SESSION['error'] = ($current_lang === 'ar') 
                ? 'خطأ: لم يتم تسجيل جميع الأسئلة. يرجى المحاولة مرة أخرى.' 
                : 'Erreur: Toutes les questions n\'ont pas été enregistrées. Veuillez réessayer.';
            header('Location: quizz_new_phases.php');
            exit;
        }
    }
    
    $stmt = $mysqli->prepare('INSERT INTO quizz_results_phases (user_id, phase, start_time, current_question_id, current_question_idx) VALUES (?, ?, UTC_TIMESTAMP(), ?, 0) ON DUPLICATE KEY UPDATE start_time = UTC_TIMESTAMP(), locked = 0, current_question_id = VALUES(current_question_id), current_question_idx = 0');
    $first_question_id = $questions[0]['id'];
    $stmt->bind_param('isi', $user_id, $phase, $first_question_id);
    $stmt->execute();
    $stmt->close();
    
    // Sauvegarder le timer dans la table de persistance
    $start_time = time();
    $stmt = $mysqli->prepare('INSERT INTO quiz_timer_sessions (user_id, phase, start_time, time_left) VALUES (?, ?, ?, 300) ON DUPLICATE KEY UPDATE start_time = VALUES(start_time), time_left = 300');
    $stmt->bind_param('isi', $user_id, $phase, $start_time);
    $stmt->execute();
    $stmt->close();
    
    // Initialiser la session pour ce quiz
    $_SESSION['quiz_phase'] = $phase;
    $_SESSION['quiz_question_idx'] = 0; // Index de 0 à 14 pour les 15 questions
    $_SESSION['quiz_answers'] = [];
    $_SESSION['quiz_started'] = true;
    $_SESSION['quiz_start_time'] = $start_time; // Timestamp de début pour le timer
    $_SESSION['quiz_current_question_id'] = $first_question_id;
    
    header('Location: quizz_new_phases.php');
    exit;
}

// Gestion de la sauvegarde du timer
if (isset($_POST['save_timer'])) {
    $phase = $_SESSION['quiz_phase'];
    $time_left = intval($_POST['time_left']);
    
    // Sauvegarder le temps restant dans la base de données
    $stmt = $mysqli->prepare('UPDATE quiz_timer_sessions SET time_left = ? WHERE user_id = ? AND phase = ?');
    $stmt->bind_param('iis', $time_left, $user_id, $phase);
    $stmt->execute();
    $stmt->close();
    
    echo json_encode(['success' => true]);
    exit;
}

// NOUVEAU SYSTÈME DE VALIDATION - AJAX (SUPPRIMÉ)
// La validation est maintenant gérée par validate_quiz_answer.php

// NOUVEAU SYSTÈME DE PASSAGE À LA QUESTION SUIVANTE - AJAX (SUPPRIMÉ)
// Le passage est maintenant géré par next_quiz_question.php

// Vérifier si le temps est écoulé pour un quiz en cours
if (isset($_SESSION['quiz_started']) && isset($_SESSION['quiz_start_time'])) {
    $time_elapsed = time() - $_SESSION['quiz_start_time'];
    if ($time_elapsed >= 300) { // 5 minutes = 300 secondes
        // Temps écoulé, terminer le quiz automatiquement
        $phase = $_SESSION['quiz_phase'];
        
        // Si la phase était "Match pour la 3e place", sauvegarder sous "Finale"
        if ($phase === 'Match pour la 3e place') {
            $phase = 'Finale';
        }
        
        $score = 0;
        foreach ($_SESSION['quiz_answers'] as $qid => $user_answer) {
            $question = getTranslatedQuestion($mysqli, $qid, $current_lang);
            if ($user_answer === $question['answer']) {
                $score += 2;
            }
        }
        
        // Sauvegarder le score (toujours sous "Finale" si c'était "Match pour la 3e place")
        $stmt = $mysqli->prepare('UPDATE quizz_results_phases SET score = ?, completed_at = UTC_TIMESTAMP(), locked = 1 WHERE user_id = ? AND phase = ?');
        $stmt->bind_param('iis', $score, $user_id, $phase);
        $stmt->execute();
        $stmt->close();
        
        // Nettoyer la session
        unset($_SESSION['quiz_phase'], $_SESSION['quiz_theme_idx'], $_SESSION['quiz_question_idx'], $_SESSION['quiz_answers'], $_SESSION['quiz_started'], $_SESSION['quiz_start_time'], $_SESSION['quiz_show_feedback'], $_SESSION['quiz_current_question_id'], $_SESSION['quiz_questions']);
        
        $_SESSION['error'] = ($current_lang === 'ar') ? 'انتهى الوقت المحدد للاختبار' : 'Le temps imparti pour le quiz est écoulé';
        header('Location: quizz_new_phases.php');
        exit;
    }
}

// Récupérer la phase sélectionnée
$selected_phase = $_SESSION['selected_phase'] ?? 'Phase de groupes';

include 'templates/header.php';
?>

<style>
.quiz-container {
    max-width: 800px;
    margin: 5px auto;
    background: linear-gradient(135deg, #23232b 80%, #ffb30022 100%);
    border-radius: 16px;
    box-shadow: 0 6px 24px rgba(0,0,0,0.3);
    padding: 10px;
    position: relative;
    overflow: hidden;
}

.quiz-container::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #ffb300, #4caf50, #2196f3, #ff9800);
    border-radius: 20px 20px 0 0;
}

.quiz-tabs {
    border-bottom: 2px solid #ffb30044;
    margin-bottom: 8px;
    position: relative;
}

.quiz-tabs::after {
    content: '';
    position: absolute;
    bottom: -2px;
    left: 0;
    right: 0;
    height: 1px;
    background: linear-gradient(90deg, transparent, #ffb300, transparent);
}

.quiz-tabs .nav-tabs {
    border-bottom: none;
    margin-bottom: 0;
    display: flex;
    justify-content: center;
    flex-wrap: wrap;
    gap: 3px;
}

.quiz-tabs .nav-tabs .nav-link {
    color: #fff;
    border: none;
    background: linear-gradient(135deg, #29293a, #1a1a22);
    border-radius: 12px 12px 0 0;
    margin-right: 0;
    padding: 8px 14px;
    font-weight: 600;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
    min-width: 100px;
    text-align: center;
    font-size: 0.9rem;
}

.quiz-tabs .nav-tabs .nav-link::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,179,0,0.3), transparent);
    transition: left 0.5s ease;
}

.quiz-tabs .nav-tabs .nav-link:hover::before {
    left: 100%;
}

.quiz-tabs .nav-tabs .nav-link:hover {
    background: linear-gradient(135deg, #ffb30022, #ffb30044);
    color: #ffb300;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(255,179,0,0.3);
}

.quiz-tabs .nav-tabs .nav-link.active {
    background: linear-gradient(135deg, #ffb300, #ffc107);
    color: #23232b;
    font-weight: 800;
    transform: translateY(-3px);
    box-shadow: 0 6px 20px rgba(255,179,0,0.4);
}

.quiz-phase-content {
    background: linear-gradient(135deg, #29293a, #1e1e2a);
    border-radius: 12px;
    padding: 12px;
    margin-top: 8px;
    border: 2px solid #ffb30044;
    position: relative;
    overflow: hidden;
}

.quiz-phase-info {
    text-align: center;
    margin-bottom: 20px;
}

.quiz-phase-info h3 {
    color: #ffb300;
    font-size: 1.3rem;
    font-weight: 900;
    margin-bottom: 15px;
}

.quiz-phase-info p {
    color: #fff;
    font-size: 1.1rem;
    margin-bottom: 15px;
}

.quiz-stats {
    display: flex;
    justify-content: space-around;
    margin-bottom: 20px;
    flex-wrap: wrap;
    gap: 15px;
}

.quiz-stat {
    background: #23232b;
    border-radius: 10px;
    padding: 15px;
    text-align: center;
    border: 2px solid #ffb30044;
    min-width: 120px;
}

.quiz-stat .stat-label {
    color: #ffb300;
    font-size: 0.9rem;
    font-weight: 600;
    margin-bottom: 5px;
}

.quiz-stat .stat-value {
    color: #fff;
    font-size: 1.2rem;
    font-weight: 800;
}

.quiz-score-display {
    background: #23232b;
    border-radius: 10px;
    padding: 20px;
    margin: 20px 0;
    text-align: center;
    border: 2px solid #ffb30044;
}

.quiz-score-display .score-value {
    color: #ffb300;
    font-size: 1.8rem;
    font-weight: 900;
    display: inline;
    margin-right: 8px;
}

.quiz-score-display .score-label {
    color: #fff;
    font-size: 1rem;
    font-weight: 600;
}

.btn {
    padding: 8px 16px;
    border-radius: 8px;
    font-weight: 700;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
    position: relative;
    overflow: hidden;
    text-transform: uppercase;
    letter-spacing: 0.3px;
    box-shadow: 0 3px 10px rgba(0,0,0,0.2);
    font-size: 0.85rem;
}

.btn::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    transition: left 0.5s ease;
}

.btn:hover::before {
    left: 100%;
}

.btn-success {
    background: linear-gradient(135deg, #4caf50, #66bb6a);
    color: #fff;
    border: 2px solid #4caf50;
}

.btn-success:hover {
    background: linear-gradient(135deg, #66bb6a, #81c784);
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(76,175,80,0.4);
}

.btn-warning {
    background: linear-gradient(135deg, #ff9800, #ffb74d);
    color: #fff;
    border: 2px solid #ff9800;
}

.btn-warning:hover {
    background: linear-gradient(135deg, #ffb74d, #ffcc80);
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(255,152,0,0.4);
}

.btn-lg {
    padding: 10px 20px;
    font-size: 0.9rem;
}

.quiz-question-container {
    background: linear-gradient(135deg, #23232b, #1e1e2a);
    border-radius: 10px;
    padding: 12px;
    margin: 8px 0;
    border: 2px solid #ffb30044;
    position: relative;
    overflow: hidden;
}

.quiz-question-container::before {
    display: none;
}

.quiz-question h4 {
    color: #fff;
    font-size: 1rem;
    font-weight: 700;
    margin-bottom: 10px;
    line-height: 1.3;
    padding-right: 25px;
}

.quiz-choices {
    display: grid;
    gap: 6px;
}

.quiz-choice {
    background: linear-gradient(135deg, #29293a, #1a1a22);
    border: 2px solid #ffb30044;
    border-radius: 8px;
    padding: 8px 10px;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.quiz-choice::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,179,0,0.1), transparent);
    transition: left 0.5s ease;
}

.quiz-choice:hover::before {
    left: 100%;
}

.quiz-choice:hover {
    background: linear-gradient(135deg, #ffb30022, #ffb30044);
    border-color: #ffb300;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(255,179,0,0.2);
}

.quiz-choice input[type="radio"] {
    display: none;
}

.quiz-choice-label {
    color: #fff;
    font-weight: 600;
    cursor: pointer;
    display: block;
    padding-left: 6px;
    position: relative;
    z-index: 1;
    font-size: 0.9rem;
}

.quiz-choice input[type="radio"]:checked + .quiz-choice-label {
    color: #ffb300;
}

.quiz-choice input[type="radio"]:checked + .quiz-choice-label.correct-answer {
    color:rgb(0, 0, 0);
}

.quiz-choice:has(input[type="radio"]:checked) {
    background: #ffb30022;
    border-color: #ffb300;
}

.quiz-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 8px;
    gap: 8px;
    flex-wrap: nowrap;
}

/* Desktop : timer, progression et score sur une ligne */
@media (min-width: 768px) {
    .quiz-header {
        display: flex;
        flex-direction: row;
        justify-content: space-between;
        align-items: center;
        gap: 8px;
    }
    
    .quiz-progress {
        flex: 1;
        margin-bottom: 0;
        margin-left: 8px;
        margin-right: 8px;
    }
}

/* Mobile : timer et score sur une ligne, progression en dessous */
@media (max-width: 767px) {
    .quiz-header {
        display: flex;
        flex-direction: row;
        justify-content: space-between;
        align-items: center;
        gap: 0;
        flex-wrap: nowrap;
    }
    
    /* Timer et score partagent l'espace équitablement sans gap */
    .quiz-header .quiz-timer-container {
        flex: 1;
        margin-right: 0;
        border-radius: 8px 0 0 8px;
    }
    
    .quiz-header .quiz-score-header {
        flex: 1;
        margin-left: 0;
        border-radius: 0 8px 8px 0;
        border-left: none;
    }
    
    /* Cacher la progression dans le header en mobile */
    .quiz-header .quiz-progress {
        display: none;
    }
    
    /* Afficher la progression mobile en dessous du header */
    .quiz-progress-mobile {
        display: block !important;
        width: 100%;
        margin-top: 8px;
        margin-bottom: 8px;
    }
}

/* Score dans le header */
.quiz-score-header {
    background: linear-gradient(135deg, #23232b, #1a1a22);
    border-radius: 8px;
    padding: 8px 12px;
    border: 2px solid #ffb30044;
    text-align: center;
    flex: 0 0 auto;
    min-width: 80px;
    position: relative;
    overflow: hidden;
}

.quiz-score-header .quiz-score-value {
    color: #ffb300;
    font-size: 1.4rem;
    font-weight: 900;
    font-family: 'Courier New', monospace;
    margin-bottom: 2px;
    text-shadow: 0 1px 3px rgba(0,0,0,0.5);
}

.quiz-score-header .quiz-score-label {
    color: #fff;
    font-size: 0.7rem;
    font-weight: 600;
    opacity: 0.8;
}

/* Desktop : timer, progression et score sur une ligne */
@media (min-width: 768px) {
    .quiz-header {
        display: flex;
        flex-direction: row;
        justify-content: space-between;
        align-items: center;
        gap: 8px;
    }
    
    .quiz-header .quiz-progress {
        flex: 1;
        margin-bottom: 0;
        margin-left: 8px;
        margin-right: 8px;
    }
    
    .quiz-progress-mobile {
        display: none !important;
    }
}

/* Mobile : timer et score sur une ligne, progression en dessous */
@media (max-width: 767px) {
    .quiz-header {
        display: flex;
        flex-direction: row;
        justify-content: space-between;
        align-items: center;
        gap: 0;
        flex-wrap: nowrap;
    }
    
    /* Timer et score partagent l'espace équitablement sans gap */
    .quiz-header .quiz-timer-container {
        flex: 1;
        margin-right: 0;
        border-radius: 8px 0 0 8px;
    }
    
    .quiz-header .quiz-score-header {
        flex: 1;
        margin-left: 0;
        border-radius: 0 8px 8px 0;
        border-left: none;
    }
    
    /* Cacher la progression dans le header en mobile */
    .quiz-header .quiz-progress {
        display: none;
    }
    
    /* Afficher la progression mobile en dessous du header */
    .quiz-progress-mobile {
        display: block !important;
        width: 100%;
        margin-top: 8px;
        margin-bottom: 8px;
    }
}

.quiz-timer-container {
    background: linear-gradient(135deg, #23232b, #1a1a22);
    border-radius: 8px;
    padding: 8px 12px;
    border: 2px solid #ffb30044;
    text-align: center;
    flex: 0 0 auto;
    min-width: 100px;
    position: relative;
    overflow: hidden;
}

.quiz-timer-container::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, transparent, rgba(255,179,0,0.1), transparent);
    animation: shimmer 2s infinite;
}

@keyframes shimmer {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

.quiz-timer {
    display: inline-block;
    position: relative;
    z-index: 1;
}

.timer-display {
    color: #ffb300;
    font-size: 1.4rem;
    font-weight: 900;
    font-family: 'Courier New', monospace;
    margin-bottom: 2px;
    text-shadow: 0 1px 3px rgba(0,0,0,0.5);
    transition: all 0.3s ease;
}

.timer-label {
    color: #fff;
    font-size: 0.7rem;
    font-weight: 600;
    opacity: 0.8;
}

.quiz-theme-display {
    background: linear-gradient(135deg, #ffb30022, #ffb30044);
    border-radius: 8px;
    padding: 6px 10px;
    border: 2px solid #ffb300;
    text-align: center;
    flex: 1;
    position: relative;
    overflow: hidden;
}

.quiz-theme-display::before {
    content: '🎯';
    position: absolute;
    left: 8px;
    top: 50%;
    transform: translateY(-50%);
    font-size: 1rem;
}

.quiz-theme-display h3 {
    color: #ffb300;
    font-size: 0.9rem;
    font-weight: 700;
    margin: 0;
    text-shadow: 0 1px 2px rgba(0,0,0,0.3);
}

.quiz-feedback {
    background: linear-gradient(135deg, #23232b, #1e1e2a);
    border-radius: 8px;
    padding: 8px;
    margin: 8px 0;
    border: 2px solid #ffb30044;
    position: relative;
    overflow: hidden;
}

.quiz-feedback::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, transparent, rgba(255,179,0,0.05), transparent);
    animation: feedbackShimmer 2s infinite;
}

@keyframes feedbackShimmer {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

.feedback-message {
    display: flex;
    align-items: center;
    gap: 12px;
    font-size: 1rem;
    font-weight: 700;
    padding: 15px 20px;
    border-radius: 12px;
    position: relative;
    z-index: 1;
    animation: feedbackSlide 0.6s ease-out;
    min-height: 60px;
    box-shadow: 0 6px 20px rgba(0,0,0,0.15);
    backdrop-filter: blur(10px);
}

@keyframes feedbackSlide {
    0% { transform: translateY(-20px); opacity: 0; }
    100% { transform: translateY(0); opacity: 1; }
}

@keyframes confetti {
    0% { 
        transform: translateY(-100vh) rotate(0deg); 
        opacity: 1; 
    }
    100% { 
        transform: translateY(100vh) rotate(360deg); 
        opacity: 0; 
    }
}

.feedback-message.correct {
    background: linear-gradient(135deg, #4caf50, #66bb6a, #81c784);
    color: #fff;
    border: 3px solid #2e7d32;
    box-shadow: 0 8px 25px rgba(76,175,80,0.4);
    position: relative;
    overflow: hidden;
}

.feedback-message.correct::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
    animation: shimmer 2s infinite;
}

.feedback-message.wrong {
    background: linear-gradient(135deg, #f44336, #ef5350, #e57373);
    color: #fff;
    border: 3px solid #c62828;
    box-shadow: 0 8px 25px rgba(244,67,54,0.4);
    position: relative;
    overflow: hidden;
}

.feedback-message.wrong::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    animation: shimmer 2s infinite;
}

.feedback-icon {
    font-size: 2rem;
    font-weight: 900;
    text-shadow: 0 2px 4px rgba(0,0,0,0.3);
    animation: iconPulse 0.6s ease-out;
    display: flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    border-radius: 50%;
    backdrop-filter: blur(5px);
}

@keyframes iconPulse {
    0% { transform: scale(0.5); opacity: 0; }
    50% { transform: scale(1.2); }
    100% { transform: scale(1); opacity: 1; }
}

.feedback-icon.correct {
    background: rgba(255, 255, 255, 0);
    color: #fff;
}

.feedback-icon.wrong {
    color: #fff;
}

.quiz-choice.correct-answer {
    background: #e8f5e9 !important;
    border-color: #4caf50 !important;
    border-width: 2px !important;
    color: #2e7d32 !important;
    box-shadow: 0 4px 12px rgba(76, 175, 80, 0.3) !important;
}

.quiz-choice.wrong-answer {
    background: #ffebee !important;
    border-color: #e53935 !important;
    border-width: 2px !important;
    color: #c62828 !important;
    box-shadow: 0 4px 12px rgba(244, 67, 54, 0.3) !important;
}

.quiz-choice.correct-answer .quiz-choice-label {
    color: #0d4f1c !important;
    font-weight: 800 !important;
    text-shadow: 0 1px 2px rgba(0,0,0,0.3);
}

.quiz-choice.wrong-answer .quiz-choice-label {
    color: #c62828 !important;
    font-weight: 800 !important;
}

/* Assurer que les styles persistent même après rechargement */
.quiz-choice.correct-answer,
.quiz-choice.wrong-answer {
    pointer-events: none;
    opacity: 1 !important;
}

.quiz-progress {
    background: linear-gradient(135deg, #29293a, #1e1e2a);
    border-radius: 8px;
    padding: 8px 10px;
    margin-bottom: 8px;
    border: 2px solid #ffb30044;
    position: relative;
    overflow: hidden;
}

.quiz-progress-mobile {
    background: linear-gradient(135deg, #29293a, #1e1e2a);
    border-radius: 8px;
    padding: 8px 10px;
    margin-bottom: 8px;
    border: 2px solid #ffb30044;
    position: relative;
    overflow: hidden;
    width: 100%;
    display: none; /* Caché par défaut, affiché en mobile via media query */
}

.quiz-progress-mobile::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, transparent, rgba(255,179,0,0.05), transparent);
    animation: progressShimmer 3s infinite;
}

.quiz-progress::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, transparent, rgba(255,179,0,0.05), transparent);
    animation: progressShimmer 3s infinite;
}

@keyframes progressShimmer {
    0% { transform: translateX(-100%); }
    100% { transform: translateX(100%); }
}

.progress-bar {
    background: linear-gradient(90deg, #4caf50, #ffb300, #ff9800);
    height: 4px;
    border-radius: 2px;
    transition: width 0.5s ease;
    margin-bottom: 4px;
    position: relative;
    z-index: 1;
    box-shadow: 0 1px 3px rgba(0,0,0,0.3);
}

.progress-text {
    color: #fff;
    font-weight: 600;
    font-size: 0.75rem;
    position: relative;
    z-index: 1;
    text-align: center;
}

@media (max-width: 600px) {
    .quiz-container {
        margin: 5px;
        padding: 8px;
    }
    
    .quiz-tabs .nav-tabs .nav-link {
        padding: 6px 10px;
        font-size: 0.8rem;
    }
    
    /* Header responsive : timer et score sur une ligne, progression en dessous */
    .quiz-header {
        flex-direction: row;
        align-items: center;
        justify-content: space-between;
        flex-wrap: nowrap;
        margin-bottom: 8px;
    }
    
    .quiz-timer-container {
        min-width: auto;
        flex: 0 0 auto;
    }
    
    .quiz-score-header {
        min-width: auto;
        flex: 0 0 auto;
    }
    
    /* La progression passe à la ligne suivante en mobile */
    .quiz-progress {
        width: 100%;
        margin-top: 0;
    }
    
    .quiz-stats {
        flex-direction: column;
        align-items: center;
    }
    
    /* Header responsive : timer et score sur une ligne, progression en dessous */
    .quiz-header {
        flex-direction: row;
        align-items: center;
        justify-content: space-between;
        flex-wrap: nowrap;
        gap: 8px;
    }
    
    .quiz-timer-container {
        min-width: auto;
        flex: 0 0 auto;
    }
    
    .quiz-score-header {
        min-width: auto;
        flex: 0 0 auto;
    }
    
    /* La progression passe à la ligne suivante en mobile */
    .quiz-progress {
        width: 100%;
        margin-top: 0;
    }
}

/* Animations pour les icônes de feedback */
@keyframes iconBounce {
    0% { 
        transform: scale(0) rotate(-180deg); 
        opacity: 0; 
    }
    50% { 
        transform: scale(1.2) rotate(-90deg); 
        opacity: 0.8; 
    }
    100% { 
        transform: scale(1) rotate(0deg); 
        opacity: 1; 
    }
}

@keyframes correctGlow {
    0% { 
        box-shadow: 0 4px 12px rgba(76, 175, 80, 0.3);
        transform: scale(1);
    }
    50% { 
        box-shadow: 0 8px 25px rgba(76, 175, 80, 0.6);
        transform: scale(1.02);
    }
    100% { 
        box-shadow: 0 4px 12px rgba(76, 175, 80, 0.3);
        transform: scale(1);
    }
}

.feedback-icon {
    display: inline-block;
    animation: iconBounce 0.8s cubic-bezier(0.68, -0.55, 0.265, 1.55);
    transition: all 0.3s ease;
    vertical-align: middle;
}

.feedback-icon svg {
    filter: drop-shadow(0 2px 4px rgba(0,0,0,0.2));
    transition: all 0.3s ease;
}

.feedback-icon:hover svg {
    transform: scale(1.1);
    filter: drop-shadow(0 4px 8px rgba(0,0,0,0.3));
}
</style>

<!-- Messages d'erreur et de succès -->
<?php if (isset($_SESSION['error'])): ?>
    <div class="alert alert-danger" style="max-width: 1000px; margin: 20px auto; background: #ffebee; color: #c62828; border: 2px solid #e53935; border-radius: 10px; padding: 15px; text-align: center;">
        <?= htmlspecialchars($_SESSION['error']) ?>
    </div>
    <?php unset($_SESSION['error']); ?>
<?php endif; ?>

<?php if (isset($_SESSION['success'])): ?>
    <div class="alert alert-success" style="max-width: 1000px; margin: 20px auto; background: #e8f5e9; color: #2e7d32; border: 2px solid #4caf50; border-radius: 10px; padding: 15px; text-align: center;">
        <?= htmlspecialchars($_SESSION['success']) ?>
    </div>
    <?php unset($_SESSION['success']); ?>
<?php endif; ?>

<div class="quiz-container page-quizz-new-phases">
    <ul class="nav nav-tabs" id="quizTabs" role="tablist">
        <?php foreach ($phases as $phase_key => $phase_name): ?>
        <li class="nav-item" role="presentation">
            <button class="nav-link <?= ($phase_key === $selected_phase) ? 'active' : '' ?>" 
                    id="<?= $phase_key ?>-tab" 
                    data-bs-toggle="tab" 
                    data-bs-target="#<?= $phase_key ?>" 
                    type="button" 
                    role="tab"
                    onclick="selectPhase('<?= $phase_key ?>')">
                <?= htmlspecialchars($phase_name) ?>
            </button>
        </li>
        <?php endforeach; ?>
    </ul>
    
    <div class="tab-content" id="quizTabsContent">
        <?php foreach ($phases as $phase_key => $phase_name): ?>
        <div class="tab-pane fade <?= ($phase_key === $selected_phase) ? 'show active' : '' ?>" 
             id="<?= $phase_key ?>" 
             role="tabpanel">
            
            <?php
            // Vérifier si la phase est active
            $phase_is_active = isPhaseActive($mysqli, $phase_key);
            
            if (!$phase_is_active): ?>
                <!-- Phase non disponible -->
                <div class="quiz-phase-content">
                    <div class="quiz-phase-info">
                        <h3><?= htmlspecialchars($phase_name) ?></h3>
                        <div style="text-align: center; padding: 40px 20px;">
                            <p style="font-size: 1.3rem; color: #ffb300; font-weight: bold; margin-bottom: 15px;">
                                <?= ($current_lang === 'ar') ? 'هذه المرحلة غير نشطة بعد' : 'Cette phase n\'est pas encore active' ?>
                            </p>
                            <p style="color: #fff; font-size: 1rem;">
                                <?= ($current_lang === 'ar') ? 'ستكون نشطة في بداية ' . htmlspecialchars($phase_name) : 'Elle sera active au début de la phase ' . htmlspecialchars($phase_name) ?>
                            </p>
                        </div>
                    </div>
                </div>
            <?php else:
            $quiz_state = getQuizState($mysqli, $user_id, $phase_key);
            $is_completed = $quiz_state['locked'] == 1;
            $is_started = $quiz_state['start_time'] !== null;
            ?>
            
            <?php if ($is_completed): ?>
                <!-- Quiz terminé -->
                <div class="quiz-phase-content">
                    <div class="quiz-phase-info">
                        <h3><?= htmlspecialchars($phase_name) ?></h3>
                        <p><?= ($current_lang === 'ar') ? 'تم إكمال الاختبار' : 'Quiz terminé' ?></p>
                        <div class="quiz-score-display">
                            <span class="score-value"><?= $quiz_state['score'] ?>/<?= isset($_SESSION['quiz_questions'][$phase_key]) ? count($_SESSION['quiz_questions'][$phase_key]) * 2 : 30 ?></span>
                            <span class="score-label"><?= __('points_label') ?></span>
                        </div>
                    </div>
                </div>
            <?php elseif ($is_started): ?>
                <!-- Quiz en cours -->
                <?php
                // Restaurer la session si nécessaire
                if (!isset($_SESSION['quiz_started']) || $_SESSION['quiz_phase'] !== $phase_key) {
                    $_SESSION['quiz_started'] = true;
                    $_SESSION['quiz_phase'] = $phase_key;
                }
                
                $question_idx = isset($_SESSION['quiz_question_idx']) ? $_SESSION['quiz_question_idx'] : 0;
                
                // Récupérer les 15 questions pour cette phase
                $questions = getQuestionsForPhase($mysqli, $user_id, $phase_key, $current_lang, true);
                
                 if (empty($questions)) {
                     echo '<div class="quiz-phase-content">';
                     echo '<div class="quiz-phase-info">';
                     echo '<h3>' . htmlspecialchars($phase_name) . '</h3>';
                     echo '<p style="color: #ffb300; font-size: 1.2rem; margin: 20px 0;">' . (($current_lang === 'ar') ? 'لا توجد أسئلة في قاعدة البيانات' : 'Aucune question disponible dans la base de données') . '</p>';
                     echo '</div>';
                     echo '</div>';
                 } else {
                    // Récupérer la question actuelle depuis la base de données
                    $current_question = null;
                    
                    // Vérifier si une question est déjà en cours dans la base de données
                    $stmt = $mysqli->prepare('SELECT current_question_id, current_question_idx FROM quizz_results_phases WHERE user_id = ? AND phase = ?');
                    if ($stmt) {
                        $stmt->bind_param('is', $user_id, $phase_key);
                        $stmt->execute();
                        $result = $stmt->get_result();
                        $quiz_data = $result->fetch_assoc();
                        $stmt->close();
                    } else {
                        $quiz_data = null;
                    }
                    
                    if ($quiz_data && $quiz_data['current_question_id']) {
                        // Récupérer la question sauvegardée
                        $table_name = ($current_lang === 'ar') ? 'quizz_questions_ar' : 'quizz_questions';
                        $stmt = $mysqli->prepare("SELECT * FROM {$table_name} WHERE id = ?");
                        $stmt->bind_param('i', $quiz_data['current_question_id']);
                        $stmt->execute();
                        $result = $stmt->get_result();
                        $current_question = $result->fetch_assoc();
                        $stmt->close();
                        
                        if ($current_question) {
                            // Restaurer l'index sauvegardé
                            if ($quiz_data['current_question_idx'] !== null) {
                                $_SESSION['quiz_question_idx'] = $quiz_data['current_question_idx'];
                                $question_idx = $quiz_data['current_question_idx'];
                            }
                            
                            // Mettre à jour la session avec la question actuelle
                            $_SESSION['quiz_current_question_id'] = $current_question['id'];
                        }
                    }
                    
                    // Si aucune question sauvegardée, utiliser la question de l'index actuel
                    if (!$current_question) {
                        // Vérifier que le tableau questions n'est pas vide et que l'index est valide
                        if (!empty($questions) && isset($questions[$question_idx]) && $question_idx < count($questions)) {
                            $current_question = $questions[$question_idx];
                            if ($current_question && isset($current_question['id'])) {
                                $_SESSION['quiz_current_question_id'] = $current_question['id'];
                                
                                // Sauvegarder la question actuelle et l'index dans la base de données
                                $stmt = $mysqli->prepare('UPDATE quizz_results_phases SET current_question_id = ?, current_question_idx = ? WHERE user_id = ? AND phase = ?');
                                if ($stmt) {
                                    $stmt->bind_param('iisi', $current_question['id'], $question_idx, $user_id, $phase_key);
                                    $stmt->execute();
                                    $stmt->close();
                                }
                            } else {
                                // Question invalide, terminer le quiz
                                $stmt = $mysqli->prepare('UPDATE quizz_results_phases SET locked = 1 WHERE user_id = ? AND phase = ?');
                                $stmt->bind_param('is', $user_id, $phase_key);
                                $stmt->execute();
                                $stmt->close();
                                
                                unset($_SESSION['quiz_started'], $_SESSION['quiz_phase'], $_SESSION['quiz_question_idx'], $_SESSION['quiz_answers'], $_SESSION['quiz_start_time'], $_SESSION['quiz_questions']);
                                
                                $_SESSION['error'] = ($current_lang === 'ar') ? 'خطأ في البيانات، يرجى إعادة تشغيل الاختبار' : 'Erreur dans les données, veuillez redémarrer le quiz';
                                header('Location: quizz_new_phases.php');
                                exit;
                            }
                        } else {
                            // Index invalide ou tableau vide, terminer le quiz
                            $stmt = $mysqli->prepare('UPDATE quizz_results_phases SET locked = 1 WHERE user_id = ? AND phase = ?');
                            $stmt->bind_param('is', $user_id, $phase_key);
                            $stmt->execute();
                            $stmt->close();
                            
                            unset($_SESSION['quiz_started'], $_SESSION['quiz_phase'], $_SESSION['quiz_question_idx'], $_SESSION['quiz_answers'], $_SESSION['quiz_start_time'], $_SESSION['quiz_questions']);
                            
                            $_SESSION['error'] = ($current_lang === 'ar') ? 'تم تحديث النظام، يرجى إعادة تشغيل الاختبار' : 'Le système a été mis à jour, veuillez redémarrer le quiz';
                            header('Location: quizz_new_phases.php');
                            exit;
                        }
                    }
                    
                    // Vérifier que current_question est valide avant de continuer
                    if (!$current_question || !isset($current_question['id'])) {
                        echo '<div class="quiz-phase-content">';
                        echo '<div class="quiz-phase-info">';
                        echo '<h3>' . htmlspecialchars($phase_name) . '</h3>';
                        echo '<p style="color: #ffb300; font-size: 1.2rem; margin: 20px 0;">' . (($current_lang === 'ar') ? 'خطأ في تحميل السؤال' : 'Erreur lors du chargement de la question') . '</p>';
                        echo '</div>';
                        echo '</div>';
                        continue; // Passer à la phase suivante
                    }
                    
                    $total_questions = count($questions);
                    $current_question_num = $question_idx + 1;
                    $progress_percent = ($current_question_num / $total_questions) * 100;
                    
                    // Calculer le score actuel
                    // IMPORTANT: Vérifier les deux tables (français et arabe) pour cumuler tous les points
                    $current_score = 0;
                    if (isset($_SESSION['quiz_answers'])) {
                        foreach ($_SESSION['quiz_answers'] as $qid => $user_answer) {
                            $is_correct = false;
                            
                            // Vérifier dans les deux tables (français et arabe)
                            $tables_to_check = ['quizz_questions', 'quizz_questions_ar'];
                            
                            foreach ($tables_to_check as $table) {
                                $stmt = $mysqli->prepare("SELECT answer, option1, option2, option3, option4 FROM {$table} WHERE id = ?");
                                if ($stmt) {
                                    $stmt->bind_param('i', $qid);
                                    $stmt->execute();
                                    $result = $stmt->get_result();
                                    $q = $result->fetch_assoc();
                                    $stmt->close();
                                    
                                    if ($q) {
                                        $correct_text = '';
                                        if (in_array($q['answer'], ['A', 'B', 'C', 'D'])) {
                                            switch($q['answer']) {
                                                case 'A': $correct_text = $q['option1']; break;
                                                case 'B': $correct_text = $q['option2']; break;
                                                case 'C': $correct_text = $q['option3']; break;
                                                case 'D': $correct_text = $q['option4']; break;
                                            }
                                        } else {
                                            $correct_text = $q['answer'];
                                        }
                                        
                                        // Comparaison avec trim pour être cohérent avec la validation
                                        if (trim($user_answer) === trim($correct_text)) {
                                            $is_correct = true;
                                            break; // Réponse trouvée, pas besoin de vérifier l'autre table
                                        }
                                    }
                                }
                            }
                            
                            if ($is_correct) {
                                $current_score += 2;
                            }
                        }
                    }
                    
                    // NOUVELLE LOGIQUE DE FEEDBACK - SIMPLIFIÉE
                    $show_feedback = false;
                    $user_answer = '';
                    $correct_answer = $current_question['answer'];
                    $is_correct = false;
                    
                    // Vérifier si cette question a déjà été répondue (après rechargement de page)
                    $question_already_answered = isset($_SESSION['quiz_answers'][$current_question['id']]);
                    
                    // Vérifier si on doit afficher le feedback pour cette question
                    if ($question_already_answered || 
                        (isset($_SESSION['quiz_show_feedback']) && 
                        $_SESSION['quiz_show_feedback'] && 
                        isset($_SESSION['quiz_current_question_id']) && 
                        $_SESSION['quiz_current_question_id'] == $current_question['id'])) {
                        
                        $show_feedback = true;
                        
                        // Utiliser le résultat de validation si disponible
                        if (isset($_SESSION['quiz_validation_result']) && 
                            $_SESSION['quiz_validation_result']['question_id'] == $current_question['id']) {
                            $validation_result = $_SESSION['quiz_validation_result'];
                            $user_answer = $validation_result['user_answer'];
                            $correct_answer = $validation_result['correct_answer'];
                            $is_correct = $validation_result['is_correct'];
                        } else {
                            // Fallback : récupérer la réponse depuis la session
                            if (isset($_SESSION['quiz_answers'][$current_question['id']])) {
                                $user_answer = $_SESSION['quiz_answers'][$current_question['id']];
                                
                                // Déterminer la bonne réponse
                                $correct_answer_letter = $current_question['answer'];
                                if (in_array($correct_answer_letter, ['A', 'B', 'C', 'D'])) {
                                    switch($correct_answer_letter) {
                                        case 'A': $correct_answer = $current_question['option1']; break;
                                        case 'B': $correct_answer = $current_question['option2']; break;
                                        case 'C': $correct_answer = $current_question['option3']; break;
                                        case 'D': $correct_answer = $current_question['option4']; break;
                                    }
                                } else {
                                    $correct_answer = $correct_answer_letter;
                                }
                                
                                $is_correct = (trim($user_answer) === trim($correct_answer));
                            }
                        }
                    }
                    ?>
                    <div class="quiz-phase-content">
                        <!-- Header responsive : Desktop (timer | progression | score), Mobile (timer | score puis progression) -->
                        <div class="quiz-header">
                            <div class="quiz-timer-container">
                                <div class="quiz-timer">
                                    <div class="timer-display" id="timer">10:00</div>
                                    <div class="timer-label"><?= ($current_lang === 'ar') ? 'الوقت المتبقي' : 'Temps restant' ?></div>
                                </div>
                            </div>
                            
                            <!-- Progression - dans le header en desktop, en dessous en mobile -->
                            <div class="quiz-progress">
                                <div class="progress-bar" style="width: <?= $progress_percent ?>%"></div>
                                <div class="progress-text">
                                    <?= $current_question_num ?>/<?= $total_questions ?> <?= ($current_lang === 'ar') ? 'سؤال' : __('questions') ?>
                                </div>
                            </div>
                            
                            <!-- Score - visible sur desktop et mobile -->
                            <div class="quiz-score-header">
                                <div class="quiz-score-value"><?= $current_score ?></div>
                                <div class="quiz-score-label"><?= __('points_label') ?></div>
                            </div>
                        </div>
                        
                        <!-- Progression mobile - affichée en dessous du header en mobile -->
                        <div class="quiz-progress-mobile">
                            <div class="progress-bar" style="width: <?= $progress_percent ?>%"></div>
                            <div class="progress-text">
                                <?= $current_question_num ?>/<?= $total_questions ?> <?= ($current_lang === 'ar') ? 'سؤال' : __('questions') ?>
                            </div>
                        </div>
                        
                        <div class="quiz-question-container">
                            <h4 style="color: #fff; font-size: 1.2rem; margin-bottom: 15px; padding: 10px; background: rgba(255,179,0,0.1); border-radius: 8px; border-left: 4px solid #ffb300;">
                                <?= htmlspecialchars($current_question['question']) ?>
                            </h4>
                            
                            <div id="quizForm">
                                <input type="hidden" name="question_id" value="<?= $current_question['id'] ?>">
                                
                                <div class="quiz-choices">
                                    <?php
                                    $options = [
                                        'A' => $current_question['option1'],
                                        'B' => $current_question['option2'],
                                        'C' => $current_question['option3'],
                                        'D' => $current_question['option4']
                                    ];
                                    
                                    // NOUVELLE LOGIQUE D'AFFICHAGE DES CHOIX - CORRIGÉE
                                    foreach ($options as $key => $option):
                                        // Déterminer l'état de chaque choix
                                        $choice_is_selected = false;
                                        $choice_is_correct = false;
                                        $choice_is_wrong = false;
                                        
                                        if ($show_feedback) {
                                            $choice_is_selected = ($user_answer === $option);
                                            $choice_is_correct = ($option === $correct_answer);
                                            $choice_is_wrong = $choice_is_selected && !$choice_is_correct;
                                        }
                                        
                                        $choice_class = 'quiz-choice';
                                        $label_class = 'quiz-choice-label';
                                        if ($show_feedback) {
                                            if ($choice_is_correct) {
                                                $choice_class .= ' correct-answer';
                                                $label_class .= ' correct-answer';
                                            } elseif ($choice_is_wrong) {
                                                $choice_class .= ' wrong-answer';
                                            }
                                        }
                                    ?>
                                    <div class="<?= $choice_class ?>">
                                        <input type="radio" name="answer" value="<?= htmlspecialchars($option, ENT_QUOTES, 'UTF-8') ?>" id="choice_<?= $key ?>" 
                                               <?= $choice_is_selected ? 'checked' : '' ?> 
                                               <?= ($show_feedback || $question_already_answered) ? 'disabled' : 'required' ?>>
                                        <label for="choice_<?= $key ?>" class="<?= $label_class ?>">
                                            <strong><?= $key ?>.</strong> <?= htmlspecialchars($option) ?>
                                            <?php if ($show_feedback && $choice_is_correct): ?>
                                                <span class="feedback-icon correct" style="margin-left: 8px;">
                                                    <svg width="22" height="22" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                        <defs>
                                                            <linearGradient id="correctGrad2_<?= $key ?>" x1="0%" y1="0%" x2="100%" y2="100%">
                                                                <stop offset="0%" style="stop-color:#4caf50;stop-opacity:1"></stop>
                                                                <stop offset="100%" style="stop-color:#2e7d32;stop-opacity:1"></stop>
                                                            </linearGradient>
                                                        </defs>
                                                        <circle cx="12" cy="12" r="11" fill="url(#correctGrad2_<?= $key ?>)" stroke="#1b5e20" stroke-width="1"></circle>
                                                        <path d="M8 12l2.5 2.5 5-5" stroke="white" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                    </svg>
                                                </span>
                                            <?php elseif ($show_feedback && $choice_is_wrong): ?>
                                                <span class="feedback-icon wrong" style="margin-left: 8px;">
                                                    <svg width="22" height="22" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                        <defs>
                                                            <linearGradient id="wrongGrad_<?= $key ?>" x1="0%" y1="0%" x2="100%" y2="100%">
                                                                <stop offset="0%" style="stop-color:#f44336;stop-opacity:1"></stop>
                                                                <stop offset="100%" style="stop-color:#c62828;stop-opacity:1"></stop>
                                                            </linearGradient>
                                                        </defs>
                                                        <circle cx="12" cy="12" r="11" fill="url(#wrongGrad_<?= $key ?>)" stroke="#b71c1c" stroke-width="1"></circle>
                                                        <path d="M15 9l-6 6M9 9l6 6" stroke="white" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                    </svg>
                                                </span>
                                            <?php endif; ?>
                                        </label>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                                
                                <?php if ($show_feedback && $question_already_answered): ?>
                                    <!-- Question déjà répondue : afficher le bouton "Question suivante" -->
                                    <button type="button" class="btn btn-warning btn-lg" style="margin-top: 20px;" onclick="nextQuestion()" id="nextBtn">
                                        <i class="fas fa-arrow-right"></i> <?= ($current_lang === 'ar') ? 'السؤال التالي' : 'Question suivante' ?>
                                    </button>
                                <?php else: ?>
                                    <!-- Question non répondue : afficher le bouton "Valider" -->
                                    <button type="button" class="btn btn-success btn-lg" style="margin-top: 20px;" onclick="validateAnswer()" id="validateBtn">
                                        <i class="fas fa-check"></i> <?= ($current_lang === 'ar') ? 'تحقق من الإجابة' : 'Valider la réponse' ?>
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <script>
                    // Réappliquer les styles des bonnes/mauvaises réponses au chargement de la page
                    (function() {
                        function applyAnswerStyles() {
                            var choices = document.querySelectorAll('.quiz-choice');
                            choices.forEach(function(choiceDiv) {
                                var label = choiceDiv.querySelector('.quiz-choice-label');
                                if (!label) return;
                                
                                // Vérifier si c'est une bonne réponse
                                if (choiceDiv.classList.contains('correct-answer')) {
                                    choiceDiv.style.background = '#e8f5e9';
                                    choiceDiv.style.borderColor = '#4caf50';
                                    choiceDiv.style.boxShadow = '0 4px 12px rgba(76, 175, 80, 0.3)';
                                    if (label) {
                                        label.style.color = '#0d4f1c';
                                        label.style.fontWeight = '800';
                                    }
                                }
                                
                                // Vérifier si c'est une mauvaise réponse
                                if (choiceDiv.classList.contains('wrong-answer')) {
                                    choiceDiv.style.background = '#ffebee';
                                    choiceDiv.style.borderColor = '#e53935';
                                    choiceDiv.style.boxShadow = '0 4px 12px rgba(244, 67, 54, 0.3)';
                                    if (label) {
                                        label.style.color = '#c62828';
                                        label.style.fontWeight = '800';
                                    }
                                }
                            });
                        }
                        
                        // Appliquer les styles au chargement
                        if (document.readyState === 'loading') {
                            document.addEventListener('DOMContentLoaded', applyAnswerStyles);
                        } else {
                            applyAnswerStyles();
                        }
                    })();
                    
                    // Timer simplifié et fiable - Compatible Android 7 (ES5)
                    // Approche par compteur décrémental pour meilleure compatibilité
                    var quizId = '<?= $_SESSION['user_id'] ?? 'guest' ?>_<?= $_SESSION['quiz_phase'] ?? 'default' ?>_<?= $_SESSION['quiz_start_time'] ?? time() ?>';
                    var totalTime = 300; // 5 minutes en secondes
                    var timerElement = null;
                    var currentTimeLeft = totalTime; // Compteur actuel
                    var timerInterval = null;
                    var timerStartTime = null; // Timestamp de démarrage du timer côté client
                    
                    // Temps de début depuis le serveur (timestamp Unix)
                    var serverStartTime = <?= $_SESSION['quiz_start_time'] ?? time() ?>;
                    
                    // Fonction de compatibilité pour obtenir le timestamp actuel (Android 7)
                    function getCurrentTimestamp() {
                        try {
                            if (typeof Date.now === 'function') {
                                return Math.floor(Date.now() / 1000);
                            } else {
                                return Math.floor(new Date().getTime() / 1000);
                            }
                        } catch(e) {
                            // Fallback ultime pour Android 7
                            var d = new Date();
                            return Math.floor(d.getTime() / 1000);
                        }
                    }
                    
                    // Initialiser le temps restant au démarrage
                    function initializeTimeLeft() {
                        var now = getCurrentTimestamp();
                        var elapsed = 0;
                        
                        // Vérifier que serverStartTime est valide
                        if (serverStartTime && serverStartTime > 0) {
                            elapsed = now - serverStartTime;
                        } else {
                            // Si pas de startTime, commencer à 300 secondes
                            elapsed = 0;
                        }
                        
                        currentTimeLeft = Math.max(0, totalTime - elapsed);
                        
                        // Si le temps est déjà écoulé, mettre à 0
                        if (currentTimeLeft <= 0) {
                            currentTimeLeft = 0;
                        }
                        
                        timerStartTime = now;
                        
                        if (typeof console !== 'undefined' && console.log) {
                            console.log('Timer Init - serverStartTime:', serverStartTime, 'now:', now, 'elapsed:', elapsed, 'currentTimeLeft:', currentTimeLeft, 'formatTime:', formatTime(currentTimeLeft));
                        }
                    }
                    
                    // Attendre que le DOM soit chargé pour Android 7
                    function initTimer() {
                        timerElement = document.getElementById('timer');
                        if (!timerElement) {
                            setTimeout(initTimer, 100);
                            return;
                        }
                        
                        // Initialiser le temps restant
                        initializeTimeLeft();
                        
                        // Afficher immédiatement
                        updateDisplay();
                        
                        // Démarrer l'intervalle - utiliser window.setInterval pour compatibilité Android 7
                        if (timerInterval) {
                            clearInterval(timerInterval);
                        }
                        timerInterval = window.setInterval(function() {
                            tickTimer();
                        }, 1000);
                    }
                    
                    // Fonction pour décrémenter le timer
                    function tickTimer() {
                        if (currentTimeLeft > 0) {
                            currentTimeLeft = currentTimeLeft - 1;
                        }
                        updateDisplay();
                        
                        if (currentTimeLeft <= 0) {
                            // Arrêter le timer
                            if (timerInterval) {
                                clearInterval(timerInterval);
                                timerInterval = null;
                            }
                            // Auto-submit
                            var quizForm = document.getElementById('quizForm');
                            if (quizForm) {
                                quizForm.submit();
                            }
                        }
                    }
                    
                    // Mettre à jour l'affichage
                    function updateDisplay() {
                        if (!timerElement) {
                            timerElement = document.getElementById('timer');
                            if (!timerElement) {
                                return;
                            }
                        }
                        
                        var timeString = formatTime(currentTimeLeft);
                        
                        // Forcer la mise à jour même si la valeur semble identique (Android 7)
                        timerElement.innerHTML = '';
                        timerElement.innerHTML = timeString;
                        
                        // Forcer le reflow pour Android 7
                        var forceReflow = timerElement.offsetHeight;
                        
                        if (typeof console !== 'undefined' && console.log) {
                            console.log('Timer Update - currentTimeLeft:', currentTimeLeft, 'timeString:', timeString);
                        }
                        
                        // Changer la couleur selon le temps restant
                        if (currentTimeLeft <= 60) {
                            timerElement.style.color = '#e53935';
                            timerElement.style.textShadow = '0 0 10px #e53935';
                        } else if (currentTimeLeft <= 180) {
                            timerElement.style.color = '#ff9800';
                            timerElement.style.textShadow = '0 0 8px #ff9800';
                        } else {
                            timerElement.style.color = '#ffb300';
                            timerElement.style.textShadow = '0 0 6px #ffb300';
                        }
                    }
                    
                    // Effet de pulsation pour le timer
                    function pulseTimer() {
                        if (timerElement) {
                            timerElement.style.transform = 'scale(1.1)';
                            setTimeout(function() {
                                if (timerElement) {
                                    timerElement.style.transform = 'scale(1)';
                                }
                            }, 200);
                        }
                    }
                    
                    // Fonction pour formater le temps (compatible Android 7 - ES5)
                    function formatTime(seconds) {
                        var secs = Math.max(0, Math.floor(seconds));
                        var mins = Math.floor(secs / 60);
                        secs = secs % 60;
                        var minsStr = (mins < 10) ? '0' + mins : String(mins);
                        var secsStr = (secs < 10) ? '0' + secs : String(secs);
                        return minsStr + ':' + secsStr;
                    }
                    
                    // Ajouter des effets aux choix (compatible Android 7)
                    var choices = document.querySelectorAll('.quiz-choice');
                    for (var i = 0; i < choices.length; i++) {
                        (function(choice) {
                            choice.addEventListener('click', function() {
                                choice.style.transform = 'scale(0.98)';
                                setTimeout(function() {
                                    choice.style.transform = 'scale(1)';
                                }, 100);
                            });
                        })(choices[i]);
                    }
                    
                    // Sauvegarder seulement le temps lors du reload, ne pas terminer le quiz
                    window.addEventListener('beforeunload', function() {
                        // Le timer continue de fonctionner avec le compteur décrémental
                        // Pas besoin de sauvegarder car on recalcule depuis le serveur au rechargement
                    });
                    
                    // NOUVEAU SYSTÈME DE VALIDATION AJAX
                    function validateAnswer() {
                        const selectedAnswer = document.querySelector('input[name="answer"]:checked');
                        
                        if (!selectedAnswer) {
                            alert('<?= ($current_lang === 'ar') ? 'يرجى اختيار إجابة قبل المتابعة' : 'Veuillez sélectionner une réponse avant de continuer' ?>');
                            return false;
                        }
                        
                        // Désactiver le bouton
                        const validateBtn = document.getElementById('validateBtn');
                        if (validateBtn) {
                            validateBtn.disabled = true;
                            validateBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> <?= ($current_lang === 'ar') ? 'جاري التحقق...' : 'Validation...' ?>';
                        }
                        
                        // Envoyer la requête AJAX
                        const formData = new FormData();
                        formData.append('question_id', document.querySelector('input[name="question_id"]').value);
                        formData.append('answer', selectedAnswer.value);
                        
                        fetch('validate_quiz_answer.php', {
                            method: 'POST',
                            body: formData
                        })
                        .then(response => {
                            console.log('Réponse reçue:', response.status);
                            if (!response.ok) {
                                throw new Error('Erreur HTTP: ' + response.status);
                            }
                            return response.text(); // Utiliser text() au lieu de json()
                        })
                        .then(text => {
                            try {
                                const data = JSON.parse(text);
                                
                                if (data.success) {
                                    showFeedback(data);
                                    
                                    // Vérifier si le quiz est terminé après cette validation
                                    if (data.is_quiz_completed) {
                                        // Le quiz est terminé, afficher le score final avec SweetAlert et rediriger
                                        setTimeout(() => {
                                            // Déterminer le message selon le score
                                            let message = '';
                                            if (data.final_score >= 120) {
                                                message = '<?= ($current_lang === 'ar') ? 'عمل ممتاز! 🏆' : 'Excellent travail! 🏆' ?>';
                                            } else if (data.final_score >= 90) {
                                                message = '<?= ($current_lang === 'ar') ? 'ممتاز! 👍' : 'Très bien! 👍' ?>';
                                            } else if (data.final_score >= 60) {
                                                message = '<?= ($current_lang === 'ar') ? 'أحسنت! 😊' : 'Bien joué! 😊' ?>';
                                            } else {
                                                message = '<?= ($current_lang === 'ar') ? 'استمر في التحسن! 💪' : 'Continuez à vous améliorer! 💪' ?>';
                                            }
                                            
                                            Swal.fire({
                                                title: '<?= __('quiz_finished') ?>',
                                                html: `
                                                    <div style="text-align: center;">
                                                        <div style="font-size: 3rem; margin: 20px 0;">🎉</div>
                                                        <h3 style="color: #4caf50; margin: 20px 0;"><?= ($current_lang === 'ar') ? 'تهانينا!' : 'Félicitations!' ?></h3>
                                                        <div style="background: linear-gradient(135deg, #4caf50, #66bb6a); color: white; padding: 20px; border-radius: 15px; margin: 20px 0;">
                                                            <div style="font-size: 2.5rem; font-weight: bold; margin-bottom: 10px;">${data.final_score} <?= __('points') ?></div>
                                                            <div style="font-size: 1.1rem;"><?= ($current_lang === 'ar') ? 'النتيجة النهائية من 30 نقطة' : 'Score final sur 30 points' ?></div>
                                                        </div>
                                                        <div style="color: #666; margin-top: 15px;">${message}</div>
                                                    </div>
                                                `,
                                                icon: 'success',
                                                confirmButtonText: '<?= ($current_lang === 'ar') ? 'العودة للاختبار' : 'Retour au quiz' ?>',
                                                confirmButtonColor: '#4caf50',
                                                allowOutsideClick: false,
                                                allowEscapeKey: false,
                                                showConfirmButton: true,
                                                timer: null,
                                                didOpen: () => {
                                                    // Ajouter une animation de confettis si le score est bon
                                                    if (data.final_score >= 90) {
                                                        // Animation de confettis simple
                                                        const confetti = document.createElement('div');
                                                        confetti.innerHTML = '🎊 🎉 🎊 🎉 🎊 🎉 🎊 🎉';
                                                        confetti.style.position = 'fixed';
                                                        confetti.style.top = '0';
                                                        confetti.style.left = '0';
                                                        confetti.style.width = '100%';
                                                        confetti.style.height = '100%';
                                                        confetti.style.pointerEvents = 'none';
                                                        confetti.style.fontSize = '2rem';
                                                        confetti.style.animation = 'confetti 3s ease-out forwards';
                                                        confetti.style.zIndex = '9999';
                                                        document.body.appendChild(confetti);
                                                        
                                                        // Supprimer les confettis après l'animation
                                                        setTimeout(() => {
                                                            if (confetti.parentNode) {
                                                                confetti.parentNode.removeChild(confetti);
                                                            }
                                                        }, 3000);
                                                    }
                                                }
                                            }).then((result) => {
                                                if (result.isConfirmed) {
                                                    window.location.href = 'quizz_new_phases.php';
                                                }
                                            });
                                        }, 2000); // Attendre 2 secondes pour que l'utilisateur voie le feedback
                                    }
                                } else {
                                    alert('Erreur: ' + data.error);
                                    resetValidateButton();
                                }
                            } catch (e) {
                                console.error('Erreur de parsing JSON:', e);
                                console.error('Texte reçu:', text);
                                alert('Erreur de format de réponse');
                                resetValidateButton();
                            }
                        })
                        .catch(error => {
                            console.error('Erreur de fetch:', error);
                            alert('Erreur de connexion: ' + error.message);
                            resetValidateButton();
                        });
                        
                        return false; // Empêcher la soumission du formulaire
                    }
                    
                    // Afficher le feedback
                    function showFeedback(data) {
                        // Debug: afficher les données reçues
                        console.log('Données reçues dans showFeedback:', data);
                        console.log('is_correct:', data.is_correct);
                        console.log('correct_answer:', data.correct_answer);
                        
                        // Masquer le bouton de validation
                        const validateBtn = document.getElementById('validateBtn');
                        if (validateBtn) {
                            validateBtn.style.display = 'none';
                        }
                        
                        // Désactiver tous les choix et ajouter les icônes
                        document.querySelectorAll('input[name="answer"]').forEach(input => {
                            input.disabled = true;
                            
                            // Trouver le label et le conteneur parent
                            const label = document.querySelector(`label[for="${input.id}"]`);
                            const choiceDiv = input.closest('.quiz-choice');
                            
                            if (label && choiceDiv) {
                                // Vérifier si c'est la réponse de l'utilisateur
                                const isUserAnswer = input.checked;
                                // Vérifier si c'est la bonne réponse - comparaison plus robuste
                                const userChoice = input.value.trim();
                                const correctAnswer = data.correct_answer.trim();
                                
                                // Comparaison principale
                                let isCorrectAnswer = userChoice === correctAnswer;
                                
                                // Comparaison alternative si la première échoue
                                if (!isCorrectAnswer) {
                                    // Essayer avec decodeURIComponent (pour les caractères encodés)
                                    try {
                                        const decodedUser = decodeURIComponent(userChoice);
                                        const decodedCorrect = decodeURIComponent(correctAnswer);
                                        isCorrectAnswer = decodedUser === decodedCorrect;
                                        console.log('Comparaison avec decodeURIComponent:', isCorrectAnswer);
                                    } catch (e) {
                                        console.log('Erreur decodeURIComponent:', e);
                                    }
                                }
                                
                                // Comparaison avec normalisation Unicode (pour gérer les caractères similaires)
                                if (!isCorrectAnswer) {
                                    // Normaliser les chaînes pour comparer les caractères Unicode équivalents
                                    const normalizedUser = userChoice.normalize('NFD').replace(/[\u0300-\u036f]/g, '');
                                    const normalizedCorrect = correctAnswer.normalize('NFD').replace(/[\u0300-\u036f]/g, '');
                                    isCorrectAnswer = normalizedUser === normalizedCorrect;
                                    console.log('Comparaison avec normalisation Unicode:', isCorrectAnswer);
                                }
                                
                                // Debug: afficher les valeurs de comparaison
                                console.log('Choix (trimmed):', `"${userChoice}"`);
                                console.log('Correct answer (trimmed):', `"${correctAnswer}"`);
                                console.log('isUserAnswer:', isUserAnswer);
                                console.log('isCorrectAnswer:', isCorrectAnswer);
                                console.log('Comparaison exacte:', userChoice === correctAnswer);
                                console.log('Longueur choix:', userChoice.length);
                                console.log('Longueur correct:', correctAnswer.length);
                                
                                // Ajouter l'icône appropriée et les classes CSS
                                if (isUserAnswer && isCorrectAnswer) {
                                    // Bonne réponse de l'utilisateur
                                    choiceDiv.classList.add('correct-answer');
                                    label.classList.add('correct-answer');
                                    label.innerHTML += ' <span class="feedback-icon correct" style="margin-left: 8px;">\
                                        <svg width="22" height="22" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">\
                                            <defs>\
                                                <linearGradient id="correctGrad" x1="0%" y1="0%" x2="100%" y2="100%">\
                                                    <stop offset="0%" style="stop-color:#4caf50;stop-opacity:1" />\
                                                    <stop offset="100%" style="stop-color:#2e7d32;stop-opacity:1" />\
                                                </linearGradient>\
                                                <filter id="glow">\
                                                    <feGaussianBlur stdDeviation="3" result="coloredBlur"/>\
                                                    <feMerge> \
                                                        <feMergeNode in="coloredBlur"/>\
                                                        <feMergeNode in="SourceGraphic"/>\
                                                    </feMerge>\
                                                </filter>\
                                            </defs>\
                                            <circle cx="12" cy="12" r="11" fill="url(#correctGrad)" stroke="#1b5e20" stroke-width="1" filter="url(#glow)"/>\
                                            <path d="M8 12l2.5 2.5 5-5" stroke="white" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/>\
                                        </svg>\
                                    </span>';
                                    choiceDiv.style.background = '#e8f5e9';
                                    choiceDiv.style.borderColor = '#4caf50';
                                    choiceDiv.style.borderWidth = '2px';
                                    choiceDiv.style.boxShadow = '0 4px 12px rgba(76, 175, 80, 0.3)';
                                    choiceDiv.style.animation = 'correctGlow 2s ease-in-out';
                                    if (label) {
                                        label.style.color = '#0d4f1c';
                                        label.style.fontWeight = '800';
                                    }
                                } else if (isUserAnswer && !isCorrectAnswer) {
                                    // Mauvaise réponse de l'utilisateur
                                    choiceDiv.classList.add('wrong-answer');
                                    label.classList.add('wrong-answer');
                                    label.innerHTML += ' <span class="feedback-icon wrong" style="margin-left: 8px;">\
                                        <svg width="22" height="22" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">\
                                            <defs>\
                                                <linearGradient id="wrongGrad" x1="0%" y1="0%" x2="100%" y2="100%">\
                                                    <stop offset="0%" style="stop-color:#f44336;stop-opacity:1" />\
                                                    <stop offset="100%" style="stop-color:#c62828;stop-opacity:1" />\
                                                </linearGradient>\
                                            </defs>\
                                            <circle cx="12" cy="12" r="11" fill="url(#wrongGrad)" stroke="#b71c1c" stroke-width="1"/>\
                                            <path d="M15 9l-6 6M9 9l6 6" stroke="white" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/>\
                                        </svg>\
                                    </span>';
                                    choiceDiv.style.background = '#ffebee';
                                    choiceDiv.style.borderColor = '#e53935';
                                    choiceDiv.style.borderWidth = '2px';
                                    choiceDiv.style.boxShadow = '0 4px 12px rgba(244, 67, 54, 0.3)';
                                    if (label) {
                                        label.style.color = '#c62828';
                                        label.style.fontWeight = '800';
                                    }
                                } else if (!isUserAnswer && isCorrectAnswer) {
                                    // Bonne réponse (pas sélectionnée par l'utilisateur)
                                    choiceDiv.classList.add('correct-answer');
                                    label.classList.add('correct-answer');
                                    label.innerHTML += ' <span class="feedback-icon correct" style="margin-left: 8px;">\
                                        <svg width="22" height="22" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">\
                                            <defs>\
                                                <linearGradient id="correctGrad2" x1="0%" y1="0%" x2="100%" y2="100%">\
                                                    <stop offset="0%" style="stop-color:#4caf50;stop-opacity:1" />\
                                                    <stop offset="100%" style="stop-color:#2e7d32;stop-opacity:1" />\
                                                </linearGradient>\
                                            </defs>\
                                            <circle cx="12" cy="12" r="11" fill="url(#correctGrad2)" stroke="#1b5e20" stroke-width="1"/>\
                                            <path d="M8 12l2.5 2.5 5-5" stroke="white" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/>\
                                        </svg>\
                                    </span>';
                                    choiceDiv.style.background = '#e8f5e9';
                                    choiceDiv.style.borderColor = '#4caf50';
                                    choiceDiv.style.borderWidth = '2px';
                                    choiceDiv.style.boxShadow = '0 4px 12px rgba(76, 175, 80, 0.3)';
                                    if (label) {
                                        label.style.color = '#0d4f1c';
                                        label.style.fontWeight = '800';
                                    }
                                }
                            }
                        });
                        
                        // Afficher le bouton "Question suivante" seulement si le quiz n'est pas terminé
                        if (!data.is_quiz_completed) {
                            const nextBtn = document.createElement('button');
                            nextBtn.type = 'button';
                            nextBtn.className = 'btn btn-warning btn-lg';
                            nextBtn.style.marginTop = '20px';
                            nextBtn.innerHTML = '<?= ($current_lang === 'ar') ? 'السؤال التالي' : 'Question suivante' ?>';
                            nextBtn.onclick = nextQuestion;
                            
                            // Insérer le bouton après les choix
                            const choicesDiv = document.querySelector('.quiz-choices');
                            choicesDiv.parentNode.insertBefore(nextBtn, choicesDiv.nextSibling);
                        } else {
                            // Si le quiz est terminé, afficher un message
                            const completedMsg = document.createElement('div');
                            completedMsg.className = 'alert alert-success';
                            completedMsg.style.marginTop = '20px';
                            completedMsg.style.textAlign = 'center';
                            completedMsg.innerHTML = '<strong><?= ($current_lang === 'ar') ? 'تم الانتهاء من الاختبار!' : 'Quiz terminé!' ?></strong><br><?= ($current_lang === 'ar') ? 'جاري إعادة التوجيه...' : 'Redirection en cours...' ?>';
                            
                            // Insérer le message après les choix
                            const choicesDiv = document.querySelector('.quiz-choices');
                            choicesDiv.parentNode.insertBefore(completedMsg, choicesDiv.nextSibling);
                        }
                    }
                    
                    // Passer à la question suivante
                    function nextQuestion() {
                        const formData = new FormData();
                        formData.append('next_question', '1');
                        
                        fetch('next_quiz_question.php', {
                            method: 'POST',
                            body: formData
                        })
                        .then(response => {
                            console.log('Réponse next_question:', response.status);
                            if (!response.ok) {
                                throw new Error('Erreur HTTP: ' + response.status);
                            }
                            return response.text();
                        })
                        .then(text => {
                            try {
                                const data = JSON.parse(text);
                                
                                if (data.success) {
                                    if (data.action === 'quiz_completed') {
                                        alert('Quiz terminé! Score: ' + data.score + ' points');
                                        window.location.href = 'quizz_new_phases.php';
                                    } else {
                                        // Recharger la page pour la question suivante
                                        window.location.reload();
                                    }
                                } else {
                                    alert('Erreur: ' + data.error);
                                }
                            } catch (e) {
                                console.error('Erreur de parsing JSON:', e);
                                alert('Erreur de format de réponse');
                            }
                        })
                        .catch(error => {
                            console.error('Erreur de fetch next_question:', error);
                            alert('Erreur de connexion: ' + error.message);
                        });
                    }
                    
                    // Remettre le bouton à la normale
                    function resetValidateButton() {
                        const validateBtn = document.getElementById('validateBtn');
                        if (validateBtn) {
                            validateBtn.disabled = false;
                            validateBtn.innerHTML = '<i class="fas fa-check"></i> <?= ($current_lang === 'ar') ? 'تحقق من الإجابة' : 'Valider la réponse' ?>';
                        }
                    }
                    
                    // Ajouter l'animation shake
                    const style = document.createElement('style');
                    style.textContent = `
                        @keyframes shake {
                            0%, 100% { transform: translateX(0); }
                            25% { transform: translateX(-5px); }
                            75% { transform: translateX(5px); }
                        }
                    `;
                    document.head.appendChild(style);
                    
                    // Remettre le bouton à la normale au chargement de la page
                    function resetValidateButton() {
                        const validateBtn = document.getElementById('validateBtn');
                        if (validateBtn) {
                            validateBtn.disabled = false;
                            validateBtn.innerHTML = '<i class="fas fa-check"></i> <?= ($current_lang === 'ar') ? 'تحقق من الإجابة' : 'Valider la réponse' ?>';
                        }
                    }
                    
                    // Vérifier l'état du bouton au chargement
                    document.addEventListener('DOMContentLoaded', function() {
                        resetValidateButton();
                    });
                    
                    // Démarrer le timer - Compatible Android 7
                    if (document.readyState === 'loading') {
                        document.addEventListener('DOMContentLoaded', initTimer);
                    } else {
                        initTimer();
                    }
                    
                    // Remettre le bouton à la normale
                    resetValidateButton();
                    
                    // Nettoyer le timer quand la page se ferme
                    window.addEventListener('beforeunload', function() {
                        if (timerInterval) {
                            clearInterval(timerInterval);
                        }
                    });
                    </script>
                    <?php
                }
                ?>
            <?php else: ?>
                <!-- Quiz pas encore commencé -->
                <?php
                // Vérifier s'il y a des questions pour cette phase
                $questions = getQuestionsForPhase($mysqli, $user_id, $phase_key, $current_lang, false);
                $questions_count = count($questions);
                
                if ($questions_count == 0) {
                    echo '<div class="quiz-phase-content">';
                    echo '<div class="quiz-phase-info">';
                    echo '<h3>' . htmlspecialchars($phase_name) . '</h3>';
                    echo '<p style="color: #ffb300; font-size: 1.2rem; margin: 20px 0;">' . (($current_lang === 'ar') ? 'لا توجد أسئلة في قاعدة البيانات' : 'Aucune question disponible dans la base de données') . '</p>';
                    echo '<p><small>' . (($current_lang === 'ar') ? 'يرجى إضافة أسئلة في لوحة الإدارة' : 'Veuillez ajouter des questions via l\'interface admin') . '</small></p>';
                    echo '</div>';
                    echo '</div>';
                } else {
                    ?>
                    <div class="quiz-phase-content">
                        <div class="quiz-phase-info">
                            <h3><?= htmlspecialchars($phase_name) ?></h3>
                            <p><?= ($current_lang === 'ar') ? 'شارك في الاختبار لتربح المزيد من النقاط<br><small>(15 سؤال - 2 نقاط لكل إجابة صحيحة)</small>' : 'Participez au Quiz pour gagner plus de points<br><small>(15 questions - 2 points pour chaque bonne réponse)</small>' ?></p>
                        </div>
                        
                        <form method="post" style="text-align: center; margin-top: 20px;">
                            <input type="hidden" name="phase" value="<?= $phase_key ?>">
                            <button type="submit" name="start_quiz" class="btn btn-success btn-lg">
                                <?= ($current_lang === 'ar') ? 'بدء الاختبار' : 'Commencer le quiz' ?>
                            </button>
                        </form>
                    </div>
                    <?php
                }
                ?>
            <?php endif; ?>
            <?php endif; // Fin de la vérification phase_is_active ?>
        </div>
        <?php endforeach; ?>
    </div>
</div>

<script>
function selectPhase(phaseKey) {
    const form = document.createElement('form');
    form.method = 'POST';
    form.style.display = 'none';
    
    const phaseInput = document.createElement('input');
    phaseInput.type = 'hidden';
    phaseInput.name = 'phase';
    phaseInput.value = phaseKey;
    
    const actionInput = document.createElement('input');
    actionInput.type = 'hidden';
    actionInput.name = 'select_phase';
    actionInput.value = '1';
    
    form.appendChild(phaseInput);
    form.appendChild(actionInput);
    document.body.appendChild(form);
    form.submit();
}

// Initialiser Bootstrap tabs
document.addEventListener('DOMContentLoaded', function() {
    var triggerTabList = [].slice.call(document.querySelectorAll('#quizTabs button'));
    triggerTabList.forEach(function (triggerEl) {
        var tabTrigger = new bootstrap.Tab(triggerEl);
        
        triggerEl.addEventListener('click', function (event) {
            event.preventDefault();
            tabTrigger.show();
        });
    });
});
</script>

<?php include 'templates/footer.php'; ?>
