<?php 
session_start();
$base = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\'); 

// Vérifier si un token est fourni
$token = isset($_GET['token']) ? trim($_GET['token']) : '';
if (empty($token)) {
    header('Location: ' . $base . '/welcome.php');
    exit;
}

// Nettoyer le token (supprimer les espaces et caractères invisibles)
$token = trim($token);
$token = preg_replace('/[^a-f0-9]/i', '', $token); // Ne garder que les caractères hexadécimaux

// Vérifier la longueur du token (doit être 64 caractères)
if (strlen($token) !== 64) {
    $error = 'Token invalide. Le format du token est incorrect.';
    $tokenData = null;
} else {
    // Vérifier la validité du token
    require_once 'includes/db.php';
    $error = null;
    $tokenData = null;
    
    try {
        // Étape 1: Vérifier si le token existe dans la base
        $tokenStmt = $mysqli->prepare('SELECT user_id, expires_at FROM password_reset_tokens WHERE token = ?');
        
        if (!$tokenStmt) {
            throw new Exception('Erreur de préparation de la requête: ' . $mysqli->error);
        }
        
        $tokenStmt->bind_param('s', $token);
        $tokenStmt->execute();
        $tokenResult = $tokenStmt->get_result();
        $tokenRow = $tokenResult->fetch_assoc();
        $tokenStmt->close();
        
        if (!$tokenRow) {
            $error = 'Token invalide. Ce lien de réinitialisation n\'existe pas.';
        } else {
            // Étape 2: Vérifier si le token est expiré (avec comparaison PHP pour éviter les problèmes de timezone)
            $expiresAt = new DateTime($tokenRow['expires_at']);
            $now = new DateTime();
            
            if ($expiresAt <= $now) {
                $error = 'Token expiré. Ce lien de réinitialisation a expiré. Veuillez demander un nouveau lien.';
            } else {
                // Étape 3: Vérifier si l'utilisateur existe
                $userStmt = $mysqli->prepare('SELECT id, email, username FROM Users WHERE id = ?');
                
                if (!$userStmt) {
                    throw new Exception('Erreur de préparation de la requête utilisateur: ' . $mysqli->error);
                }
                
                $userStmt->bind_param('i', $tokenRow['user_id']);
                $userStmt->execute();
                $userResult = $userStmt->get_result();
                $user = $userResult->fetch_assoc();
                $userStmt->close();
                
                if (!$user) {
                    $error = 'Token invalide. L\'utilisateur associé à ce token n\'existe plus.';
                } else {
                    // Tout est OK, récupérer les données complètes
                    $tokenData = [
                        'user_id' => $user['id'],
                        'email' => $user['email'],
                        'username' => $user['username'],
                        'expires_at' => $tokenRow['expires_at']
                    ];
                }
            }
        }
    } catch (Exception $e) {
        $error = 'Erreur lors de la vérification du token. Veuillez réessayer.';
        error_log('Reset password error: ' . $e->getMessage());
    }
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Réinitialisation du mot de passe - CAN 2025</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="<?= $base ?>/assets/style.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #00915a 0%, #006d44 100%);
            min-height: 100vh;
            font-family: 'Arial', sans-serif;
        }
        
        .reset-container {
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .reset-card {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
            max-width: 500px;
            width: 100%;
        }
        
        .reset-header {
            background: linear-gradient(135deg, #00915a 0%, #006d44 100%);
            color: white;
            padding: 40px 30px;
            text-align: center;
        }
        
        .reset-body {
            padding: 40px 30px;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 15px;
            font-size: 16px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #00915a;
            box-shadow: 0 0 0 0.2rem rgba(0, 145, 90, 0.25);
        }
        
        .btn-reset {
            background: linear-gradient(135deg, #ab2024 0%, #8b1a1a 100%);
            border: none;
            border-radius: 10px;
            padding: 15px 30px;
            font-weight: bold;
            font-size: 16px;
            color: white;
            transition: all 0.3s ease;
            width: 100%;
        }
        
        .btn-reset:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(171, 32, 36, 0.4);
            color: white;
        }
        
        .btn-back {
            background: #6c757d;
            border: none;
            border-radius: 10px;
            padding: 10px 20px;
            color: white;
            text-decoration: none;
            display: inline-block;
            transition: all 0.3s ease;
        }
        
        .btn-back:hover {
            background: #5a6268;
            color: white;
            text-decoration: none;
            transform: translateY(-1px);
        }
        
        .input-group .btn {
            border: 2px solid #e9ecef;
            border-left: none;
            border-radius: 0 10px 10px 0;
        }
        
        .input-group .form-control {
            border-radius: 10px 0 0 10px;
        }
        
        .progress {
            height: 8px;
            border-radius: 4px;
            background-color: rgba(0,0,0,0.1);
        }
        
        .progress-bar {
            border-radius: 4px;
            transition: width 0.3s ease;
        }
        
        .error-card {
            background: #f8f9fa;
            border-left: 4px solid #dc3545;
        }
        
        .success-icon {
            color: #28a745;
            font-size: 3rem;
        }
        
        .error-icon {
            color: #dc3545;
            font-size: 3rem;
        }
        
        @media (max-width: 576px) {
            .reset-container {
                padding: 10px;
            }
            
            .reset-header, .reset-body {
                padding: 30px 20px;
            }
        }
    </style>
</head>
<body>
    <div class="reset-container">
        <div class="reset-card">
            <?php if ($error): ?>
            <!-- Message d'erreur -->
            <div class="reset-header">
                <i class="fas fa-exclamation-triangle error-icon mb-3"></i>
                <h3 class="mb-0">Erreur</h3>
            </div>
            <div class="reset-body error-card">
                <p class="text-muted mb-4"><?= htmlspecialchars($error) ?></p>
                <div class="d-flex gap-2">
                    <a href="<?= $base ?>/welcome.php" class="btn-back">
                        <i class="fas fa-arrow-left me-2"></i>Retour à la connexion
                    </a>
                    <a href="<?= $base ?>/welcome.php?forgot=1" class="btn-back" style="background: #00915a;">
                        <i class="fas fa-key me-2"></i>Demander un nouveau lien
                    </a>
                </div>
            </div>
            <?php else: ?>
            <!-- Formulaire de réinitialisation -->
            <div class="reset-header">
                <i class="fas fa-key success-icon mb-3"></i>
                <h3 class="mb-2">Réinitialisation du mot de passe</h3>
                <p class="mb-0 opacity-75">Bonjour <strong><?= htmlspecialchars($tokenData['username']) ?></strong></p>
            </div>
            
            <div class="reset-body">
                <form id="resetPasswordForm">
                    <input type="hidden" id="token" value="<?= htmlspecialchars($token) ?>">
                    
                    <!-- Nouveau mot de passe -->
                    <div class="mb-4">
                        <label for="newPassword" class="form-label fw-bold text-dark mb-2">
                            <i class="fas fa-lock me-2 text-primary"></i>Nouveau mot de passe
                        </label>
                        <div class="input-group">
                            <input type="password" class="form-control" id="newPassword" name="newPassword" required minlength="6">
                            <button class="btn btn-outline-secondary" type="button" id="toggleNewPassword">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                        <div class="form-text text-muted mt-2">
                            <i class="fas fa-info-circle me-1"></i>Minimum 6 caractères
                        </div>
                    </div>
                    
                    <!-- Confirmation du mot de passe -->
                    <div class="mb-4">
                        <label for="confirmPassword" class="form-label fw-bold text-dark mb-2">
                            <i class="fas fa-lock me-2 text-primary"></i>Confirmer le mot de passe
                        </label>
                        <div class="input-group">
                            <input type="password" class="form-control" id="confirmPassword" name="confirmPassword" required minlength="6">
                            <button class="btn btn-outline-secondary" type="button" id="toggleConfirmPassword">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                    </div>
                    
                    <!-- Option de force du mot de passe -->
                    <div class="mb-4">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="showPasswordStrength" style="transform: scale(1.2);">
                            <label class="form-check-label text-dark fw-medium ms-2" for="showPasswordStrength">
                                <i class="fas fa-chart-line me-2 text-info"></i>Afficher la force du mot de passe
                            </label>
                        </div>
                    </div>
                    
                    <!-- Indicateur de force du mot de passe -->
                    <div id="passwordStrength" class="mb-4" style="display: none;">
                        <div class="progress mb-2">
                            <div class="progress-bar" role="progressbar" style="width: 0%"></div>
                        </div>
                        <small class="text-muted" id="strengthText">
                            <i class="fas fa-info-circle me-1"></i>Saisissez un mot de passe
                        </small>
                    </div>
                    
                    <!-- Bouton de soumission -->
                    <div class="d-grid">
                        <button type="submit" class="btn-reset">
                            <i class="fas fa-save me-2"></i>Réinitialiser le mot de passe
                        </button>
                    </div>
                </form>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="<?= $base ?>/assets/app.js"></script>
    <script>
    const base = "<?= $base ?>";

    // Vérifier si le formulaire existe avant d'ajouter les événements
    const resetPasswordForm = document.getElementById('resetPasswordForm');
    
    if (resetPasswordForm) {
        // Fonction pour basculer la visibilité du mot de passe
        function togglePasswordVisibility(inputId, buttonId) {
            const input = document.getElementById(inputId);
            const button = document.getElementById(buttonId);
            const icon = button.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }

        // Événements pour basculer la visibilité
        const toggleNewPassword = document.getElementById('toggleNewPassword');
        if (toggleNewPassword) {
            toggleNewPassword.addEventListener('click', function() {
                togglePasswordVisibility('newPassword', 'toggleNewPassword');
            });
        }

        const toggleConfirmPassword = document.getElementById('toggleConfirmPassword');
        if (toggleConfirmPassword) {
            toggleConfirmPassword.addEventListener('click', function() {
                togglePasswordVisibility('confirmPassword', 'toggleConfirmPassword');
            });
        }

        // Fonction pour calculer la force du mot de passe
        function calculatePasswordStrength(password) {
            let strength = 0;
            let feedback = [];
            
            if (password.length >= 6) strength += 1;
            else feedback.push('Au moins 6 caractères');
            
            if (password.length >= 8) strength += 1;
            if (password.length >= 12) strength += 1;
            
            if (/[a-z]/.test(password)) strength += 1;
            else feedback.push('Lettres minuscules');
            
            if (/[A-Z]/.test(password)) strength += 1;
            else feedback.push('Lettres majuscules');
            
            if (/[0-9]/.test(password)) strength += 1;
            else feedback.push('Chiffres');
            
            if (/[^A-Za-z0-9]/.test(password)) strength += 1;
            else feedback.push('Caractères spéciaux');
            
            return { strength: Math.min(strength, 5), feedback: feedback };
        }

        // Événement pour afficher la force du mot de passe
        const showPasswordStrength = document.getElementById('showPasswordStrength');
        if (showPasswordStrength) {
            showPasswordStrength.addEventListener('change', function() {
                const strengthDiv = document.getElementById('passwordStrength');
                if (this.checked) {
                    strengthDiv.style.display = 'block';
                    updatePasswordStrength();
                } else {
                    strengthDiv.style.display = 'none';
                }
            });
        }

        // Fonction pour mettre à jour la force du mot de passe
        function updatePasswordStrength() {
            const password = document.getElementById('newPassword').value;
            const strength = calculatePasswordStrength(password);
            const progressBar = document.querySelector('.progress-bar');
            const strengthText = document.getElementById('strengthText');
            
            if (!progressBar || !strengthText) return;
            
            const percentage = (strength.strength / 5) * 100;
            progressBar.style.width = percentage + '%';
            
            let color = 'bg-danger';
            let text = 'Très faible';
            
            if (strength.strength >= 2) {
                color = 'bg-warning';
                text = 'Faible';
            }
            if (strength.strength >= 3) {
                color = 'bg-info';
                text = 'Moyen';
            }
            if (strength.strength >= 4) {
                color = 'bg-success';
                text = 'Fort';
            }
            if (strength.strength >= 5) {
                text = 'Très fort';
            }
            
            progressBar.className = 'progress-bar ' + color;
            strengthText.textContent = text + (strength.feedback.length > 0 ? ' - Améliorer : ' + strength.feedback.join(', ') : '');
        }

        // Événement pour mettre à jour la force en temps réel
        const newPasswordInput = document.getElementById('newPassword');
        if (newPasswordInput) {
            newPasswordInput.addEventListener('input', function() {
                const showPasswordStrengthCheckbox = document.getElementById('showPasswordStrength');
                if (showPasswordStrengthCheckbox && showPasswordStrengthCheckbox.checked) {
                    updatePasswordStrength();
                }
            });
        }

        // Validation du formulaire
        resetPasswordForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const newPassword = document.getElementById('newPassword').value.trim();
            const confirmPassword = document.getElementById('confirmPassword').value.trim();
            const token = document.getElementById('token').value.trim();
            
            // Validation côté client
            if (newPassword.length < 6) {
                if (typeof showToast === 'function') {
                    showToast('Le mot de passe doit contenir au moins 6 caractères.', 'danger');
                } else {
                    alert('Le mot de passe doit contenir au moins 6 caractères.');
                }
                return;
            }
            
            if (newPassword !== confirmPassword) {
                if (typeof showToast === 'function') {
                    showToast('Les mots de passe ne correspondent pas.', 'danger');
                } else {
                    alert('Les mots de passe ne correspondent pas.');
                }
                return;
            }
            
            // Afficher un loader
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Réinitialisation en cours...';
            
            // Envoyer la requête
            fetch(base + '/ajax/reset_password.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'token=' + encodeURIComponent(token) + 
                      '&newPassword=' + encodeURIComponent(newPassword) + 
                      '&confirmPassword=' + encodeURIComponent(confirmPassword)
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error('Erreur réseau');
                }
                return response.json();
            })
            .then(data => {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
                
                if (data.success) {
                    Swal.fire({
                        icon: 'success',
                        title: 'Mot de passe réinitialisé !',
                        text: data.message,
                        confirmButtonText: 'Se connecter',
                        confirmButtonColor: '#00915a',
                        allowOutsideClick: false
                    }).then(() => {
                        window.location = base + '/welcome.php';
                    });
                } else {
                    if (typeof showToast === 'function') {
                        showToast(data.message || 'Une erreur est survenue.', 'danger');
                    } else {
                        alert(data.message || 'Une erreur est survenue.');
                    }
                }
            })
            .catch(error => {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
                if (typeof showToast === 'function') {
                    showToast('Erreur de connexion au serveur. Veuillez réessayer.', 'danger');
                } else {
                    alert('Erreur de connexion au serveur. Veuillez réessayer.');
                }
                console.error('Error:', error);
            });
        });
    }
    </script>
</body>
</html>
