<?php
// Nouveau système de validation des réponses de quiz - AJAX
require_once 'includes/db.php';
require_once 'includes/language.php';

// Démarrer la session si elle n'est pas déjà démarrée
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Vérifier que c'est une requête POST
if (!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Méthode non autorisée']);
    exit;
}

// Vérifier que l'utilisateur est connecté
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Non connecté']);
    exit;
}

// Vérifier que le quiz est démarré
if (!isset($_SESSION['quiz_started']) || !$_SESSION['quiz_started']) {
    http_response_code(400);
    echo json_encode(['error' => 'Quiz non démarré']);
    exit;
}

// Récupérer les données POST
$question_id = intval($_POST['question_id'] ?? 0);
$user_answer = trim($_POST['answer'] ?? '');

// Validation des données
if ($question_id <= 0 || empty($user_answer)) {
    http_response_code(400);
    echo json_encode(['error' => 'Données invalides']);
    exit;
}

try {
    // Récupérer la langue courante
    $current_lang = getCurrentLanguage();
    $table_name = ($current_lang === 'ar') ? 'quizz_questions_ar' : 'quizz_questions';
    
    // Récupérer la question depuis la base de données (table selon la langue)
    $stmt = $mysqli->prepare("SELECT * FROM {$table_name} WHERE id = ?");
    $stmt->bind_param('i', $question_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $question = $result->fetch_assoc();
    $stmt->close();
    
    if (!$question) {
        http_response_code(404);
        echo json_encode(['error' => 'Question introuvable']);
        exit;
    }
    
    // Déterminer si la réponse est correcte
    $correct_answer_letter = $question['answer']; // A, B, C, D ou texte direct
    $correct_answer_text = '';
    
    // Vérifier si c'est une lettre (A, B, C, D) ou du texte direct
    if (in_array($correct_answer_letter, ['A', 'B', 'C', 'D'])) {
        // Convertir la lettre en texte de l'option
        switch($correct_answer_letter) {
            case 'A':
                $correct_answer_text = $question['option1'];
                break;
            case 'B':
                $correct_answer_text = $question['option2'];
                break;
            case 'C':
                $correct_answer_text = $question['option3'];
                break;
            case 'D':
                $correct_answer_text = $question['option4'];
                break;
        }
    } else {
        // C'est déjà du texte direct
        $correct_answer_text = $correct_answer_letter;
    }
    
    // Utiliser une comparaison plus robuste (trim + insensible à la casse)
    $is_correct = (trim($user_answer) === trim($correct_answer_text));
    $points = $is_correct ? 2 : 0;
    
    // Sauvegarder la réponse dans la session
    if (!isset($_SESSION['quiz_answers'])) {
        $_SESSION['quiz_answers'] = [];
    }
    $_SESSION['quiz_answers'][$question_id] = $user_answer;
    
    // Note: Les questions sont déjà enregistrées dans quiz_answered_questions au démarrage du quiz
    // Il n'est pas nécessaire de les réenregistrer ici lors de la validation de chaque réponse
    // Cela évite les doublons et garde la table dédiée uniquement au suivi des questions posées
    
    // Vérifier si c'est la dernière question
    $question_idx = isset($_SESSION['quiz_question_idx']) ? $_SESSION['quiz_question_idx'] : 0;
    $phase = $_SESSION['quiz_phase'];
    
    // Déterminer le nombre total de questions pour cette phase
    $total_questions = 15; // Par défaut
    if (isset($_SESSION['quiz_questions'][$phase])) {
        $total_questions = count($_SESSION['quiz_questions'][$phase]);
    }
    
    $is_quiz_completed = ($question_idx >= ($total_questions - 1)); // Dernière question (indices 0 à total-1)
    
    // Mettre à jour le score partiel en base de données
    // IMPORTANT: Vérifier les deux tables (français et arabe) pour cumuler tous les points
    $current_score = 0;
    foreach ($_SESSION['quiz_answers'] as $qid => $answer) {
        $is_correct = false;
        
        // Vérifier dans les deux tables (français et arabe)
        $tables_to_check = ['quizz_questions', 'quizz_questions_ar'];
        
        foreach ($tables_to_check as $table) {
            $stmt = $mysqli->prepare("SELECT answer, option1, option2, option3, option4 FROM {$table} WHERE id = ?");
            if ($stmt) {
                $stmt->bind_param('i', $qid);
                $stmt->execute();
                $result = $stmt->get_result();
                $q = $result->fetch_assoc();
                $stmt->close();
                
                if ($q) {
                    $correct_text = '';
                    
                    // Vérifier si c'est une lettre (A, B, C, D) ou du texte direct
                    if (in_array($q['answer'], ['A', 'B', 'C', 'D'])) {
                        switch($q['answer']) {
                            case 'A': $correct_text = $q['option1']; break;
                            case 'B': $correct_text = $q['option2']; break;
                            case 'C': $correct_text = $q['option3']; break;
                            case 'D': $correct_text = $q['option4']; break;
                        }
                    } else {
                        // C'est déjà du texte direct
                        $correct_text = $q['answer'];
                    }
                    
                    // Comparaison avec trim pour être cohérent avec la validation
                    if (trim($answer) === trim($correct_text)) {
                        $is_correct = true;
                        break; // Réponse trouvée, pas besoin de vérifier l'autre table
                    }
                }
            }
        }
        
        if ($is_correct) {
            $current_score += 2;
        }
    }
    
    // Si la phase était "Match pour la 3e place", utiliser "Finale" (car combinées)
    $phase = $_SESSION['quiz_phase'];
    if ($phase === 'Match pour la 3e place') {
        $phase = 'Finale';
    }
    
    // Mettre à jour le score en base
    $stmt = $mysqli->prepare('UPDATE quizz_results_phases SET score = ? WHERE user_id = ? AND phase = ?');
    $stmt->bind_param('iis', $current_score, $_SESSION['user_id'], $phase);
    $stmt->execute();
    $stmt->close();
    
    // Marquer que cette question a été répondue en mettant à jour current_question_id dans la base
    // Cela permet de détecter lors du rechargement que la question a déjà été répondue
    $stmt = $mysqli->prepare('UPDATE quizz_results_phases SET current_question_id = ? WHERE user_id = ? AND phase = ?');
    $stmt->bind_param('iis', $question_id, $_SESSION['user_id'], $phase);
    $stmt->execute();
    $stmt->close();
    
    // Si c'est la dernière question, terminer le quiz
    if ($is_quiz_completed) {
        // Note: Les questions sont déjà enregistrées individuellement lors de chaque validation
        // Pas besoin de les réenregistrer ici pour éviter les doublons
        
        // Marquer le quiz comme terminé
        $stmt = $mysqli->prepare('UPDATE quizz_results_phases SET completed_at = UTC_TIMESTAMP(), locked = 1 WHERE user_id = ? AND phase = ?');
        $stmt->bind_param('is', $_SESSION['user_id'], $phase);
        $stmt->execute();
        $stmt->close();
        
        // Nettoyer la session
        unset($_SESSION['quiz_started']);
        unset($_SESSION['quiz_phase']);
        unset($_SESSION['quiz_question_idx']);
        unset($_SESSION['quiz_answers']);
        unset($_SESSION['quiz_start_time']);
        unset($_SESSION['quiz_questions']);
        unset($_SESSION['quiz_current_question_id']);
    }
    
    // Sauvegarder le résultat de validation
    $_SESSION['quiz_validation_result'] = [
        'question_id' => $question_id,
        'user_answer' => $user_answer,
        'correct_answer' => $correct_answer_text,
        'correct_answer_letter' => $correct_answer_letter,
        'is_correct' => $is_correct,
        'points' => $points
    ];
    
    // Retourner la réponse JSON
    echo json_encode([
        'success' => true,
        'question_id' => $question_id,
        'user_answer' => $user_answer,
        'correct_answer' => $correct_answer_text,
        'correct_answer_letter' => $correct_answer_letter,
        'is_correct' => $is_correct,
        'points' => $points,
        'is_quiz_completed' => $is_quiz_completed,
        'final_score' => $current_score,
        'message' => $is_correct ? 
            'Bonne réponse! +2 points' :
            'Mauvaise réponse. La bonne réponse est: ' . $correct_answer_text
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Erreur serveur: ' . $e->getMessage()]);
}

$mysqli->close();
?>
